#!/bin/sh
#
# Copyright 1998-2003 Microsoft Corporation -- All Rights Reserved.
#
# $Revision: 1.28 $
# $Date: 2003/12/07 12:06:46 $

#

#
# This is the main part of the shell script.
#
main() {
    initialize
    step1                               # setup environment
    step2                               # untar
    $FPDIR/set_default_perms.sh         # Run the external permissions script.
    step3                               # upgrade/install
    
    echo 
    echo "Installation completed!  Exiting..." 
    
    exit 0
}

# main install steps
step1()
{
    echo 
    echo "Step 1.  Setting Up Installation Environment" 
    echo 
    
    checkuser      || error        # Check to make sure we are root
    banner         || error        # Display header/copyright info
    usrlocalexists || error        # Check to make sure that /usr/local exists
    checkforaccess || error        # Check for RWX access to /usr/local
    wheretoinstall || error        # Find out where to install the extensions
    license                        # print license and get confirmation
}

step2()
{
    echo 
    echo "Step 2.  Untarring the Extensions and Checking Protections" 
    echo 
    
    untarext       || error        # Untar the extensions
}

step3()
{
    echo 
    echo "Step 3.  Upgrading/Installing the extensions" 
    echo 
    
    getapachevers           || error   # Find out which version of Apache
    migrateoldconfig        || error   # Migrate old frontpage.cnf (if any)
    change_server           || error   # upgrade httpd
    not_stub_apache         || error   # Check if we're debugging this script
    upgradeexistingservers  || error   # Check to see if servers need upgrading
    upgrade="yes"
    chownexistingservers    || error   # Now chown the webs
    handlelanguage          || error   # configure some global settings
    installrootweb          || error   # Install the root web
    installnewsubwebs $PORT || error   # Install new servers
    installvirtualwebs      || error   # Install any virtual webs
}

################################################################################
# basic utility functions
#

myprompt()
{
    answer="|"
    until echo $1 | grep $answer >/dev/null
    do
        $echo "${2} [${3}]?  ${nnl}" 
        read answer
        if [ "$3" != "" ] && [ "$answer" = "" ]
        then
            answer=$3
        fi
    done
}

getpassword()
{
    prompt=$1
    PASSWORD=""
    until [ "$PASSWORD" != "" ]
    do
        $echo "$prompt:  ${nnl}" 
        stty -echo
        read password1
        stty echo
        echo
        $echo "Re-enter $prompt:  ${nnl}" 
        stty -echo
        read password2
        stty echo
        echo 
        if [ "$password1" = "$password2" ]
        then
            PASSWORD=$password1
        fi
    done
    
    return 1
}

getHomedir()
{
    user=$1
    shell=`egrep "csh|ksh|bash" /etc/shells | head -1`
    homedir=`$shell -f -c 'echo ~'$user`
}

# Print an error message and exit the program.
error()
{
    echo 
    echo "Exiting due to an error!  Please fix the error and try again." 
    echo 
    
    exit 1
}

# Print a useful error message and exit the program.
die() {
    echo "Aborting, could not successfully $*"
    exit 1
}

die1() {
    echo "$*"
    exit 1
}

initialize()
{
    VERSION="5.0"
    OLDPATH=$PATH
    PATH=".:/bin:/usr/bin:/sbin:/usr/sbin:/usr/ucb:/etc:/usr/bsd"
    INSTALLDIRDEFAULT="/usr/local/frontpage"
    NEWHTTPDNEW="/usr/local/frontpage/version${VERSION}/apache-fp/httpd"
    NEWMODFPSO1="/usr/local/frontpage/version${VERSION}/apache-fp/mod_frontpage.c"
    NEWMOFPSO2="/usr/local/frontpage/version${VERSION}/apache2/mod_frontpage.c"
    DEFAULTHTTPD="/usr/local/apache/sbin/httpd"
    FPDIR="/usr/local/frontpage/version${VERSION}"
    APACHE_2_0_VERSION="2.0.40"
    unset MORE

    case "`echo 'x\c'`" in
       'x\c')   echo="echo -n"    nnl= ;;      #BSD
       x)       echo="echo"       nnl="\c" ;;  # Sys V
       *)       echo "$0 quitting:  Can't set up echo." ; exit 1 ;;
    esac
    
    TAB="	"
    system=`uname -a`
    
    case "$system" in
        OSF1*)              machine="alpha" ;;
        Linux*ppc*)         machine="linuxppc" ;;
        Linux*s390*)        machine="linuxs390" ;;
        Linux*)             machine="linux" ;;
        HP-UX*)             machine="hp700" ;;
        AIX*)               machine="rs6000" ;;
        IRIX*)              machine="sgi" ;;
        SunOS*5.*sun4*)     machine="solaris" ;;
        SunOS*5.*i386*)     machine="solarisx86" ;;
        BSD/OS*)            machine="bsdi" ;;
        FreeBSD*)           machine="freebsd" ;;
        *)              echo "ERROR:  Unsupported platform!  Uname is $system."
                        return 1
                         ;;
    esac

    case "$machine" in 
        sunos) functions="function toupper(CHAR) {
                        UPPER = \"ABCDEFGHIJKLMNOPQRSTUVWXYZ\";
                        LOWER = \"abcdefghijklmnopqrstuvwxyz\";
                        if (POS = index(LOWER,CHAR))
                            RETVAL = substr(UPPER,POS,1);
                        else 
                            RETVAL = CHAR;
                        
                        return RETVAL 
                     }
                     
                     function tolower(CHAR) {
                        UPPER = \"ABCDEFGHIJKLMNOPQRSTUVWXYZ\";
                        LOWER = \"abcdefghijklmnopqrstuvwxyz\";
                        if (POS = index(UPPER,CHAR))
                            RETVAL = substr(LOWER,POS,1);
                        else 
                            RETVAL = CHAR;
                        
                        return RETVAL 
                     }"
            ;;
         *) functions="" 
            ;;
    esac

    awk=nawk
    lsg="ls -ld"
    psea="ps -ea"
    fgrep=fgrep
    strings=cat 
    
    case "$machine" in
         alpha) strings="strings -a"
                ;;
          bsdi) awk=awk
                ;;
       freebsd) awk=awk
                strings="strings -a"
                ;;
         hp700) awk=awk
                ;;
        linux*) awk=awk
                psea="ps ea"
                strings="strings -a"
                ;;
        rs6000) strings="strings -a"
                ;;
           sgi) awk=nawk 
                lsg="ls -ld"
                ;;
       solaris) awk=nawk
                lsg="ls -ld"
                strings="strings -a"
                ;;
    solarisx86) awk=nawk
                lsg="ls -ld"
                ;;
             *)
                ;;
    esac
}

# Compare two version numbers, including all sub-versions (such as 5.0.2.1234)
# The result is stored in the third argument, which must be a name suitable
# for a shell environment variable.  This variable will be set to a string,
# either GT, EQ, or LT, as appropriate.
compareVersions()
{
    Version1=$1
    Version2=$2
    RetName=$3
    eval ${RetName}=`(echo $Version1 ; echo $Version2) | $awk -F . '
            {if (NR==1) {A=$1;B=$2;C=$3;D=$4; next}}
            {if ((A > $1) ||
                ((A == $1) && ((B > $2) ||
                ((B == $2) && ((C > $3) ||
                ((C == $3) && (D > $4))))))) {print "GT";next}}
            {if ((A == $1) && (B == $2) && 
                 (C == $3) && (D == $4)) {print "EQ";next}}
            {print "LT"}'`
}

################################################################################
# Step 1: Setup/Verify Environment
#

banner()
{
cat <<EOF

 fp_install.sh
 
 Revision$Revision: 1.28 $dummy
 Date$Date: 2003/12/07 12:06:46 $dummy
 
This script will step the user through upgrading existing and installing 
new servers and webs.  As with any software installation, a backup should be
done before continuing.  It is recommended that the FrontPage installation
directory, server configuration file directory, and all web content be 
backed up before continuing with this installation.
 
EOF

   myprompt 'yYnN' "Are you satisfied with your backup of the system (y/n)" "N"
   if [ $answer = n ] || [ $answer = N ]
   then
      exit 0
   fi
   echo
   
   return 0
}

#
# The following functions do some initial checks that we are who we need to be
# and have the directories and permissions on the directories that we need
#
checkuser()
{
    # Make sure we are running as root.
    
    whoami=`whoami 2>/dev/null` || whoami=`/usr/bin/id | sed -e ' s/).*//; s/^.*(//;'`
    retval=0
    
    echo "Setting umask 002" 
    umask 002
    
    if [ $whoami != "root" ]
    then
        echo "ERROR:  Logged in as $whoami.  Must be root to do this installation!"
        retval=1
    else
        echo "Logged in as root." 
    fi
    
    return $retval
}

usrlocalexists()
{
    # Now lets make sure that /usr/local exists.
    
    localdir="/usr/local"
    prot=755
    retval=0
    
    if [ ! -d "$localdir" ]
    then
        myprompt 'yYnN' "$localdir does not exist.  Would you like to create it (y/n)" "Y"
        if [ $answer = y ] || [ $answer = Y ]
        then
            echo "Creating $localdir" 
            if mkdir "$localdir"
            then
                echo "Directory $localdir has been created." 
                if chmod "$prot" "$localdir"
                then
                    echo "Directory $localdir chmoded to $prot." 
                else
                    echo "ERROR:  Unable to chmod $localdir to $prot." 
                    retval=1
                fi
            else
                echo "ERROR:  Unable to create $localdir!" 
                retval=1
            fi
        else
            echo "ERROR:  Directory $localdir must exist!" 
            retval=1
        fi
    else
        if [ ! -d "$localdir" ]
        then
            echo "ERROR:  $localdir exists but is not a directory!" 
            retval=1
        else
            echo "Directory $localdir exists." 
        fi
    fi
    
    return $retval
}

checkforaccess()
{
    # Make sure that we have the correct access to /usr/local
    
    retval=0
    
    # Check to make sure that we have read access to the directory
    comma=""
    if [ ! -r /usr/local ]
    then
        noaccess="Read"
        comma=", "
    fi
    
    # Check to make sure that we have write access to the directory
    if [ ! -w /usr/local ]
    then
        noaccess="$noaccess${comma}Write"
        comma=", "
    fi
    
    # Check to make sure that we have execute access to the directory
    if [ ! -x /usr/local ]
    then
        noaccess="$noaccess${comma}Execute"
    fi
    
    if  [ "$noaccess" != "" ]
    then
        echo "ERROR:  Root does not have $noaccess access to /usr/local!" 
        retval=1
    else
        echo "Root has necessary access to /usr/local." 
    fi
    
    return $retval
}

#
# Find out where to install the extensions and create directory and/or link
# as necessary.
#
wheretoinstall()
{
    retval=0
    
    cat <<EOF

Where would you like to install the FrontPage Server Extensions.  If
you select a location other than /usr/local/frontpage/ then a symbolic
link will be created from /usr/local/frontpage/ to the location that
is chosen.

EOF
    $echo "FrontPage Server Extensions directory [/usr/local/frontpage/]:  ${nnl}"
    read installdir
    
    if [ "$installdir" = "" ]
    then
        installdir=$INSTALLDIRDEFAULT
    fi
    # Normalize the name
    installdir=`dirname $installdir`/`basename $installdir`

    if [ ! -d "$installdir" ]
    then
        echo "Creating $installdir" 
        if mkdir "$installdir"
        then
            echo "Directory $installdir has been created." 
            if chmod "$prot" "$installdir"
            then
                echo "Directory $installdir chmoded to $prot." 
            else
                echo "ERROR:  Unable to chmod $installdir to $prot." 
                retval=1
            fi
        else
            echo "ERROR:  Unable to create $installdir!" 
            retval=1
        fi
    else
        echo "WARNING:  Directory $installdir already exists." 
        echo "Installation will overwrite existing files." 
        echo 
        
        myprompt 'yYnN' "Continue the installation (y/n)" "N"
        echo 
        if [ $answer = n ] || [ $answer = N ]
        then
            exit 0
        fi
    fi
    
    if [ "$installdir" != "/usr/local/frontpage" ]
    then
        if [ ! -d "/usr/local/frontpage" ]
        then
            if ln -s "$installdir" "/usr/local/frontpage"
            then
                echo "Snapped link /usr/local/frontpage --> $installdir"
            else
                echo "ERROR:  Unable to  snap link /usr/local/frontpage --> $installdir!" 
                retval=1
            fi
        else
            echo "ERROR:  Unable to snap link /usr/local/frontpage --> $installdir, /usr/local/frontpage already exists!" 
            retval=1
        fi
    fi
    
    return $retval 
}

license() 
{
    untarlicense || error
    echo "Please read the following license agreement (which can be found"
    echo "in $licensefile"
    echo
    $echo "Hit enter to continue${nnl}" 
    read continue
    more $licensefile
    echo "The above agreeement is installed in $licensefile)."
    myprompt 'yYnN' "Do you accept the terms of the license agreement (y/n)" "N"
    if [ $answer != "y" -a $answer != "Y" ] 
    then
       exit 
    fi
}

untarlicense()
{
    # This is derived from untarext
    retval=0
    vtfile="fp50.$machine.tar"
    vtfilelocation="`pwd`/"
    getextfilename $vtfilelocation $vtfile || return 1
    olddir=`pwd`
    if [ "$installdir" != "/usr/local/frontpage" ]
    then
        cd $installdir
        licensefile="./version${VERSION}/LICENSE.TXT"
    else
        cd /usr/local
        licensefile="frontpage/version${VERSION}/LICENSE.TXT"
    fi
    if [ -f "$licensefile" ] 
    then
      n=1
      while [ -f "${licensefile}.bak.$n" ] 
      do
        n=`expr $n + 1` 
      done
      mv $licensefile ${licensefile}.bak.$n
    fi
    case $fullvtfile in 
     *tar) tar -xf $fullvtfile frontpage/version${VERSION}/LICENSE.TXT ;;
       *Z) zcat $fullvtfile | tar -xf - frontpage/version${VERSION}/LICENSE.TXT 
           ;;
      *gz) zcat=""
           while [ "$zcat" = "" ] || [ ! -f "$zcat" ]
           do
               $echo "Where is the zcat which can uncompress gz files? ${nnl}"
               read zcat
               base=`basename zcat`
               case $base in
                  zcat) ;; 
                     *) zcat=`dirname $zcat`/`basename $zcat`/zcat ;;
               esac
           done
           $zcat $fullvtfile | tar -xf - frontpage/version${VERSION}/LICENSE.TXT
           zcatok=1
           ;;
        *) echo "ERROR:  Unknown file type: $fullvtfile"
           return 1
           ;;
    esac
    if [ "$installdir" != "/usr/local/frontpage" ]
    then
        mv frontpage/* .
        rm -rf frontpage
        licensefile=`pwd`/version${VERSION}/LICENSE.TXT
    else
        licensefile=`pwd`/frontpage/version${VERSION}/LICENSE.TXT
    fi
    cd $olddir
    return $retval
}

#
# Find out if this is an Apache 1.3 or Apache 2.0 installation
#
getapachevers()
{
    $echo "Which version of Apache will you be using? (1.3 or 2.0) ${nnl}"
    read ApacheVers
    while [ "$ApacheVers" != "1.3" -a "$ApacheVers" != "2.0" ]
    do
      $echo "Please enter either '1.3' or '2.0'"
      $echo "Which version of Apache will you be using? (1.3 or 2.0) ${nnl}"
      read ApacheVers
    done

    if [ "$ApacheVers" = "1.3" ]
    then
        promptstring="Where is the current apache daemon located: [$HttpdFile] ${nnl}"
    else
        promptstring="Where is the current apache control file (apachectl) located: [$HttpdFile] ${nnl}"
    fi
       
    echo
    HttpdFile=""
    while ( [ "$HttpdFile" = "" ] || [ ! -x $HttpdFile ] )
    do
        HttpdFile=$DEFAULTHTTPD
        $echo $promptstring
        read location
        if [ "$location" != "" ]
        then
	    HttpdFile=$location
        fi

	#
	# We don't change the 1.3 behavior, but 2.0 requires apachectl, so
	# we double check
	if [ "$ApacheVers" = "2.0" -a `basename $HttpdFile` != "apachectl" ]
	then
	    echo "Warning: The Apache control file you entered is not named 'apachectl'"
	    myprompt 'yYnN' "Are you sure this is correct? (y/n)" "Y"

	    if [ "$answer" != "y" -a "$answer" != "Y" ]
	    then
		# Reset it
		HttpdFile=""
	    fi
	fi

	if [ "$ApacheVers" = "1.3"  ]
	then
	   if [ ! -x "$HttpdFile" -o ! -f "$HttpdFile" ]
	   then
	    HttpdFile=""
	   fi
	fi

    done
    if [ "$ApacheVers" = "1.3" ] 
    then
        getapachevers1
    else
	getapachevers2
    fi
}

getapachevers1()
{
    # Get the installed Apache Version
    installedApacheVers=`$HttpdFile -v | sed 's%^.*Apache/\([0-9\.]*\).*%\1%' | head -1`

    echo "Currently running Apache/${installedApacheVers}"
}

getapachevers2()
{
    installedApacheVers=`$HttpdFile -v  start | sed 's%^.*Apache/\([0-9\.]*\).*%\1%' | head -1`
    echo "Currently running Apache/${installedApacheVers}"
}

###############################################################################
#
# Step 2: Untar the extensions into the installation directory and check
#         permissions
#
untarext()
{
    retval=0
    
    if [ -d "${installdir}/version${VERSION}/exes" ]
    then
        echo "Version $VERSION FrontPage Server Extensions found."
        myprompt 'yYnN' "Would you like to overwrite? (y/n)" "Y"
        echo 
        if [ $answer = n ] || [ $answer = N ]
        then
            echo "No need to un-tar the extensions.  Continuing..."
            return $retval
        else
            echo "Looking for tar file..."
        fi
    else
        echo "Version $VERSION FrontPage Server Extensions not found."
        echo "Looking for tar file..."
    fi
    
    vtfile="fp50.$machine.tar"
    echo "Platform is $machine." 
    
    vtfilelocation="`pwd`/"
    
    if test \! -f $fullvtfile ; then
       getextfilename $vtfilelocation $vtfile || return 1
    fi
    
    olddir=`pwd`

    if [ "$installdir" != "/usr/local/frontpage" ]
    then
        cd $installdir
    else
        cd /usr/local
    fi
    case $fullvtfile in 
     *tar) echo "Untarring file $fullvtfile into /usr/local..." 
           tar -xf $fullvtfile ;;
       *Z) echo "Uncompressing/Untarring file $fullvtfile into /usr/local..."
           zcat $fullvtfile | tar -xf - 
           ;;
      *gz) if [ "$zcatok" != "1" ]
           then
               zcat=""
           fi
           while [ "$zcat" = "" ] || [ ! -f "$zcat" ]
           do
               $echo "Where is the zcat which can uncompress gz files? ${nnl}"
               read zcat
               base=`basename zcat`
               case $base in
                  zcat) ;; 
                     *) zcat=`dirname $zcat`/`basename $zcat`/zcat ;;
               esac
           done
           echo "Uncompressing/Untarring file $fullvtfile into /usr/local..."
           $zcat $fullvtfile | tar -xf -
           ;;
        *) echo "ERROR:  Unknown file type: $fullvtfile"
           return 1
           ;;
    esac
    
    if [ "$installdir" != "/usr/local/frontpage" ]
    then
        cp -r frontpage/* .
        rm -rf frontpage
    fi
    
    cd $olddir
    
    return $retval
}

# get the name of the tarball file
# used by untarext
getextfilename()
{
    vtfilelocation=$1
    vtfile=$2
    fullvtfile="${vtfilelocation}${vtfile}"
    if [ ! -f "$fullvtfile" ]
    then
        if [ -f "${fullvtfile}.Z" ]
        then
            fullvtfile="${fullvtfile}.Z"       
        else
            if [ -f "${fullvtfile}.gz" ]
            then
                fullvtfile="${fullvtfile}.gz"
            else
                if [ -f "${fullvtfile}.z" ]
                then
                    echo "Renaming ${vtfile}.z to ${vtfile}.Z"
                    mv "${fullvtfile}.z" "${fullvtfile}.Z"
                    fullvtfile="${fullvtfile}.Z"
                else
                    echo "Cannot find the FrontPage Server Extensions tar file in ${vtfilelocation}." 
                    $echo "Which directory is the file located in (X to cancel)?  ${nnl}" 
                    read vtfilelocation
                    if [ "$vtfilelocation" = "X" ] || [ "$vtfilelocation" = "x" ]
                    then
                        return 1
                    else
                        vtfilelocation=`echo $vtfilelocation |sed -e 's/\/$//'`
                        
                        if [ ! -d "$vtfilelocation" ]
                        then
                            vtfilelocation=`dirname $vtfilelocation`
                        fi
                        getextfilename $vtfilelocation/ $vtfile
                    fi
                fi
            fi
        fi
    fi
}

###############################################################################
#
# Step 3: Upgrade/Install extensions
#

#
# Look for older versions of frontpage.cnf, and migrate them into the current
#
migrateoldconfig()
{
    retval=0
    
    echo "Checking for previous versions of FrontPage Server Extensions to upgrade..." 
    
    upgrade=""
    for file in ${installdir}/version*/frontpage.cnf
    do
        if [ $file != ${installdir}/version${VERSION}/frontpage.cnf ]
        then
            upgrade="$upgrade $file"
        fi
    done
    
    if [ "$upgrade" != "" ]
    then
        upgrade=`echo $upgrade | sed -e 's/.*version//g' -e 's/\/frontpage\.cnf//g'`
        echo "You have an older version of FrontPage Server Extensions installed ($upgrade).";
        myprompt 'yYnN' "Would you like to migrate these settings to version $VERSION now (y/n)" "Y"
        
        if [ $answer = n ] || [ $answer = N ] ; then
            upgrade=""
	    retval=1
            return $retval;
        fi
    else
        echo "You have no previous versions of FrontPage Server Extensions installed."
        echo "There are no settings to migrate!"
        return $retval;
    fi
    echo
    
    for ver in $upgrade ; do
        mergefiles ${installdir}/version${ver}/frontpage.cnf $FPDIR/frontpage.cnf $FPDIR/frontpage.cnf
    done

    # mergefiles mucks w/ owner/group/mode, so we must reset them here
    chown bin $FPDIR/frontpage.cnf || die chown frontpage.cnf
    chgrp bin $FPDIR/frontpage.cnf || die chgrp frontpage.cnf
    chmod 664 $FPDIR/frontpage.cnf || die chmod frontpage.cnf
    
    return $retval
}

# keys in "in1" replace keys in "in2".  keys in only one file are maintained
mergefiles()
{
    in1=$1
    in2=$2
    out=$3
    outreal=$out
    if [ $out = $in1 -o $out = $in2 ]
    then
        out=${outreal}.new
    fi
    
    cat $in1 > $out
    for key in `sed -e 's/:.*//g' $in2` ; do
        if [ `grep -i -c "^$key:" $in1` = 0 ]
        then
            grep -i "^$key:" $in2 >> $out
        fi
    done
    
    if [ $out != $outreal ]
    then
        mv -f $out $outreal
    fi
}

# Are we running against a test stub Apache?  We only do that to facilitate
# QA of the change_server code.
not_stub_apache() {
    #First check for a test stub Apache.  We use the test stub to 
    #test the fp_install.sh code
    isStubApache=""
    if [ "$ApacheVers" = "1.3" ] 
    then
      isStubApache=`$HttpdFile -v | grep "FrontPage Test Stub Apache Server"`
    else
      isStubApache=`$HttpdFile -v start | grep "FrontPage Test Stub Apache Server"`
    fi
    if [ -n "$isStubApache" ]
    then
      echo "Found stub Apache used for testing the first part of the install"
      echo "Exiting test run after change_server processing."
      exit
    fi
}

# Check to see if servers need upgrading
upgradeexistingservers()
{
    retval=0
    
    echo "Checking for existing FrontPage web servers to upgrade..." 
    
    upgrade="none"
    for file in ${installdir}/*.cnf
    do
        if [ "$file" = "${installdir}/*.cnf" ]
        then
            echo "No existing FrontPage web servers found to upgrade." 
            return $retval
        fi
        
        cat <<EOF
 
Existing web servers were found; do you want to upgrade them now?
(If you answer "no", you can safely run this script again and answer "yes"
to upgrade your servers later.)

EOF
        myprompt 'yYnN' "Upgrade now (y/n)" "Y"
        
        if [ $answer = n ] || [ $answer = N ]
        then
            upgrade="no"
            echo "For details on how to upgrade servers manually, please see" 
            echo "the Server Extension Resource Kit (SERK), located at"
            echo "http://www.microsoft.com/frontpage"
            echo
            return $retval
        else
            upgrade="yes"
        fi
        break
    done

    cat <<EOF
 
The file ${installdir}/version${VERSION}/upgrade_results.txt will
contain Success/Fail status for the upgrades.  When the upgrade is
complete you should examine this file to make sure that all of the
upgrades completed successfully.

EOF
    $echo "Hit enter to continue${nnl}" 
    read continue
    echo
    
    echo "All existing servers will now be upgraded:" 
    echo " " 
    
    createdate=`date`
    cat >${installdir}/version${VERSION}/upgrade_results.txt <<EOF 
#
# Server Upgrade Results
#
# Automatically generated by fp_install.sh on $createdate
#
EOF
    
    for weconfigfile in ${installdir}/*.cnf
    do
        echo 
        echo "Upgrading using configuration file:  "$weconfigfile 
        if verifywebserver $weconfigfile
        then
            if upgradeserver $weconfigfile
            then
                echo "Upgrade Successful  $weconfigfile" >> ${FPDIR}/upgrade_results.txt
            else
                echo "Upgrade Failed      $weconfigfile" >> ${FPDIR}/upgrade_results.txt
                echo "ERROR:  Server upgrade failed!"
                echo "Continuing with next server." 
                $echo "Hit enter to continue${nnl}" 
                read continue
            fi
        else
            echo "Cancelling upgrade..."
            echo "Upgrade Failed      $weconfigfile" >> ${FPDIR}/upgrade_results.txt
        fi
    done
    
    weconfigfile=""
    
    return $retval
}

# examine weX.cnf file, make sure it is a patched apache server
verifywebserver()
{
    weconfigfile="$1"
    
    conf=`basename $weconfigfile`
    webport=`echo  $conf | sed -e '
        s/:/:/
        tmulti
        s/.cnf$//
        s/.*[^0-9]//
        q
        :multi
        s/.cnf$//'`
    
    port=$webport
    
    echo "Verifying web server configuration..."
    
    configfile=`grep -i "^serverconfig:" $weconfigfile | sed -e '
                    s/serverconfig://g
                    s/fakeconf.*\///'`
    if [ ! -f "$configfile" ]
    then
        echo "$configfile does not exist."
        return 1
    fi

    servertype=`grep -i "^servertype:" $weconfigfile | sed -e 's/servertype://g'`
    configfiledir=`dirname $configfile`"/"

    if [ $servertype != "apache-fp" ]
    then
        echo "This version of FrontPage Server Extensions does not suppport"
        echo "$servertype servers.  We only support patched apache servers."
        echo "For more information about supported servers, please see" 
        echo "the Server Extension Resource Kit (SERK), located at"
        echo "http://www.microsoft.com/frontpage"
        return 1
    fi

    getdocroot $weconfigfile ||
    {
        echo "ERROR:  Unable to get DocumentRoot/UserDir" 
        return 1
    }

    getHttpDirective $configfile AccessConfig $port
    if [ "$param" != "" ]
    then
        case $param in
            /dev/null) accessconffile=$configfile
                ;;
            /*) accessconffile=$param  # fully qualified pathname
                ;;
            *)  file=`basename $param`
                accessconffile="${configfiledir}${file}"
                ;;
        esac
    else
        accessconffile="${configfiledir}access.conf"
        if [ ! -f $accessconffile ]
        then
            accessconffile=$configfile
        fi
    fi
    if [ "$accessconffile" != "" -a "$accessconffile" != "/dev/null" -a ! -f "$accessconffile" ]
    then
        echo "$accessconffile does not exist."
        return 1
    fi

    servicesfile=${docroot}"/_vti_pvt/services.cnf"
    
    if [ ! -f "$servicesfile" ]
    then
        echo "There are no services to upgrade for this web."
        return 1
    fi
    
    return 0
}

# Go through all webs on all servers and chown the content appropriately
chownexistingservers()
{
    retval=0
    
    if [ "$upgrade" = "yes" ]
    then
        echo
        echo "Preparing to chown webs..." 
        cat <<EOF

All requested upgrades to the new FrontPage Server Extensions have taken place.
The next step is to chown the web in order to guarantee that the extensions
will work properly.  At this point you have two options:

   1. This script will prompt you interactively for an owner and group of
      each web and then perform the chown.  If you do not have a lot of 
      webs you might want to choose this option.
      
   2. This script will generate a script, which you can edit to fill in the
      owner and group for each web, to run at a later date.  If you have a
      large number of webs you might want to choose this option.
      
EOF
        myprompt 'yYnNsS' "Would you like interactive prompting for each webs owner/group (y/n)" "Y"
        if [ $answer = s ] || [ $answer = S ]
        then
            return $retval
        fi
        if [ $answer = y ] || [ $answer = Y ]
        then
            chownwebs 
        else
            generatechownscript $file  ||
            (
                echo "ERROR: Server chown failed! Continuing with next server."
                $echo "Hit enter to continue${nnl}" 
                read continue
            )
        fi
        return $retval
    fi
}

# Go through all webs on a given server and chown the content appropriately
chownwebs()
{
    retval=0
    
    for weconfigfile in ${installdir}/*.cnf
    do
        if grep Failed ${installdir}/version${VERSION}/upgrade_results.txt | grep ${weconfigfile} > /dev/null
        then
            echo
            echo "Upgrade of ${weconfigfile} failed."
            echo "See ${installdir}/version${VERSION}/upgrade_results.txt" 
            echo "Skipping chown..."
        else
            servertype=`grep -i "^servertype:" $weconfigfile|sed -e 's/servertype://g'`
            if [ "$servertype" != "apache-fp" ]
            then
                echo "Skipping non patched-apache web $weconfigfile"
            else
                chownWeb $weconfigfile
            fi
        fi
    done
    weconfigfile=""
    return $retval
}

# chown the content of a given web/subweb
chownWeb()
{
    weconfigfile=$1
    
    conf=`basename $weconfigfile`
    webport=`echo  $conf | sed -e '
        s/:/:/
        tmulti
        s/.cnf$//
        s/.*[^0-9]//
        q
        :multi
        s/.cnf$//'`
    
    port=$webport
    
    echo 
    echo "Processing webs in port $webport..."
    echo 
    
    configfile=`grep -i "^serverconfig:" $weconfigfile | sed -e '
                    s/serverconfig://g
                    s/fakeconf.*\///'`
    configfiledir=`dirname $configfile`"/"
    
    getdocroot $weconfigfile ||
    {
        echo "ERROR:  Unable to get DocumentRoot/UserDir"
        return 1
    }
    
    servicesfile=${docroot}"/_vti_pvt/services.cnf"
    
    # map our $servicesfile to STDIN so we can use "read" on it...
    exec 4<&0
    exec <$servicesfile
    n=0
    while read service
    do
        echo
        if [ $service = "/" ]
        then
            webname=""
            webtext="root web"
            web=""
        else
            webname="$service"
            webtext="$service"
            web="-w $webname"
        fi
        exec 5<&0
        exec 0<&4
        
        getdocroot $weconfigfile ||
        {
            echo "ERROR:  Unable to get DocumentRoot/UserDir"
            return 1
        }

        # Figure out who owns (or should own) this web.  Prompt if we can't
        # figure it out ourselves.
        case $service in
           /~*) owner=`echo $service | sed -e 's/\///'`
                webowner=`echo $service | sed -e 's/\/~//'`
                getHomedir $webowner
                
                if [ -d "${homedir}/${userdir}" ]
                then
                    echo "Web ${webtext} on port ${webport} will be owned by ${webowner}"
                    defwebgroup=`$lsg ${homedir}/${userdir} | $awk ' { print $4}'`
                    exists=0
                else
                    exists="${homedir}/${userdir}"
                fi
             ;;
            *)  if [ -d "${docroot}/${service}" ]
                then
                    defwebowner=`$lsg ${docroot}${service} | $awk ' { print $3}'`
                    defwebgroup=`$lsg ${docroot}${service} | $awk ' { print $4}'`
                    webowner=""
                    until [ "$webowner" != "" ]
                    do
                        $echo "Who should own web ${webtext} on port ${webport} [${defwebowner}]:  ${nnl}"
                        read webowner
                        if [ "$webowner" = "" ]
                        then
                            webowner=$defwebowner
                        fi
                    done
                    exists=0
                else
                    exists="${docroot}/${service}"
                fi
             ;;
        esac

        # We have the owner, now let's find the group
        if [ "$exists" = "0" ]
        then
            webgroup=""
            until [ "$webgroup" != "" ]
            do
                $echo "What should the group for web ${webtext} on port ${webport} be [${defwebgroup}]:  ${nnl}"
                read webgroup
                if [ "$webgroup" = "" ]
                then
                    webgroup=$defwebgroup
                fi
            done
            
            case $webport in 
               *:*) multihost="-m "`echo $webport | cut -d: -f1`
                    multiport="-p "`echo $webport | cut -d: -f2`
                    ;;
                 *) multihost=""
                    multiport="-p $webport"
                    ;;
            esac
            
	    if [ $httpdmajor -eq 2 ]
	    then
	        echo "Using Apache V2.0"
	        fpversionoption="apache-2.0"
	    else
	        echo "Using Apache V1"
	        fpversionoption="apache-fp"
	    fi
            $FPDIR/bin/owsadm.exe -o chown $multiport $multihost $web -xUser $webowner -xGroup $webgroup -t $fpversionoption ||
            {
                echo 
                echo "ERROR: Unable to chown web ${webtext} in port ${webport}"
                $echo "Hit enter to continue${nnl}" 
                read continue
            }
        else
            echo "ERROR:  web $service - $exists does not exist!  Skipping to next web."  
        fi
            exec 0<&5
    done
    exec <&4
}

#
# Generate a script fp_chown.sh that will call "owsadm -o chown" for each web,
# Used when the user says they do not want to be prompted interactively for
# each web
# 
generatechownscript()
{
    retval=0
    scriptout="${FPDIR}/fp_chown.sh"
    createdate=`date`
    
    cat <<EOF
 
A script will be generated in ${scriptout} 
which you can edit to chown your webs at a future date.  You will 
need to edit the script and set the owner and group for each web 
before running the script.
 
EOF
    
    cat >$scriptout <<EOF 
#! /bin/sh
#
#
# Copyright 1996-2001 Microsoft Corporation -- All Rights Reserved.
#
#          Automatically generated by fp_install.sh
#
# Automatically generated by fp_install.sh on $createdate
#
# You will need to edit this script before running it.  Change each 
# <OWNER> and <GROUP> to reflect the ownership/group that you want 
# set for the web.
#
# Example:  80 /testweb webowner webgroup
#

VERSION="5.0"

chown_web()
{
    port=\$1
    webname=\$2
    webowner=\$3
    webgroup=\$4
    
    if [ "\$webowner" = "<OWNER>" ] || [ "\$webgroup" = "<GROUP>" ]
    then
        echo "WARNING:  Owner/Group not specified for web \$webname on \$port."
        echo "Skipping to next web..."
    else
        if [ "\$webname" != "/" ]
        then
            webname=\`echo \$webname | sed -e 's%^/%%g'\`
            webtext="\$webname"
            web="-w \$webname"
        else
            webtext="root web"
            web=""
        fi
        case \${port} in
           *:*) multihost="-m "\`echo \${port} | cut -d: -f1\`
                multiport="-p "\`echo \${port} | cut -d: -f2\`
                ;;
             *) multihost=""
                multiport="-p \${port}"
                ;;
        esac
        echo
        echo "Chowning web \${webtext} in port \${port} to owner \${webowner} group \${webgroup}"
        /usr/local/frontpage/version${VERSION}/bin/owsadm.exe -o chown \$multiport \$multihost \$web  -xUser \$webowner -xGroup \$webgroup -t $fpversionoption 
    fi
}
 
while read port webname webowner webgroup
do
    chown_web \$port \$webname \$webowner \$webgroup
done <<ENDCHOWN
EOF
    
    for weconfigfile in ${installdir}/*.cnf
    do
        if grep Failed ${installdir}/version${VERSION}/upgrade_results.txt | grep ${weconfigfile} > /dev/null
        then 
            echo
            echo "Upgrade of ${weconfigfile} failed."
            echo "See ${installdir}/version${VERSION}/upgrade_results.txt" 
            echo "Skipping chown..."
        else
            addChownWeb $weconfigfile
        fi
    done
    
    echo "ENDCHOWN" >>$scriptout
    
    chmod 764 $scriptout
    
    weconfigfile=""
    return $retval
}

# Very similar to chownWeb(), but stick results into a file for fp_chown.sh
addChownWeb()
{
    weconfigfile=$1
    
    echo 
    echo "Processing  $weconfigfile"
    conf=`basename $weconfigfile`
    webport=`echo  $conf | sed -e '
        s/:/:/
        tmulti
        s/.cnf$//
        s/.*[^0-9]//
        q
        :multi
        s/.cnf$//'`
    
    port=$webport
    
    echo "Adding webs in port ${webport} to chown script..."
    servertype=`grep -i "^servertype:" $weconfigfile|sed -e 's/servertype://g'`
    configfile=`grep -i "^serverconfig:" $weconfigfile | sed -e '
                s/serverconfig://g
                s/fakeconf.*\///'`
    configfiledir=`dirname $configfile`"/"
    getdocroot $weconfigfile ||
    {
        echo "ERROR:  Unable to get DocumentRoot/UserDir"
        return 1
    }
    
    servicesfile="${docroot}/_vti_pvt/services.cnf"
    if [ -r $servicesfile ]
    then
        exec 4<&0
        exec <$servicesfile
        n=0

        while read service
        do 
            if [ $service = "/" ]
            then
                webtext="root web"
            else
                webtext="$service"
            fi
            
            case $service in
               /~*) owner=`echo $service | sed -e 's/\///'`
                    webowner=`echo $service | sed -e 's/\/~//'`
                    getHomedir $webowner
                    
                    if [ -d "${homedir}/${userdir}" ]
                    then
                        webgroup=`$lsg ${homedir}/${userdir} | $awk ' { print $4}'`
                        exists=0
                    else
                        exists="${homedir}/${userdir}"
                    fi
                ;;
                 *) if [ -d "${docroot}/${service}" ]
                    then
                        webowner=`$lsg ${docroot}${service} | $awk ' { print $3}'`
                        webgroup=`$lsg ${docroot}${service} | $awk ' { print $4}'`
                        exists="0"
                    else
                        exists="${docroot}/${service}"
                    fi
                    ;;
            esac
            
            if [ "$exists" = "0" ]
            then
                echo "web ${webtext}"
                echo "$webport $service $webowner $webgroup" >> $scriptout
            else
                echo "ERROR:  web ${webtext}- Path $exists does not exist!  Skipping to next web."  
            fi
        done
        exec <&4
    else
        echo "WARNING:  Unable to read $servicesfile!" 
        echo "Skipping to next port."
    fi
}

# upgrade a server
upgradeserver()
{
    retval=0
    weconfigfile="$1"
    bindir=$installdir'/version'$VERSION'/bin'
    conf=`basename $weconfigfile`
    webport=`echo  $conf | sed -e '
        s/:/:/
        tmulti
        s/.cnf$//
        s/.*[^0-9]//
        q
        :multi
        s/.cnf$//'`
    
    echo "Upgrading server "$webport

    case $webport in
       *:*) multihost="-m "`echo $webport | cut -d: -f1`
            multiport="-p "`echo $webport | cut -d: -f2`
            ;;
         *) multihost=""
            multiport="-p $webport"
            ;;
    esac
    
    # docroot is set by getdocroot which is called from verifywebserver which
    # is called before upgradeserver
    webowner=`$lsg ${docroot} | $awk ' { print $3}'`
    xuser="-xu $webowner"
    
    $bindir/owsadm.exe -o upgrade $multiport $multihost $xuser -t $fpversionoption 
    
    return $retval
}

getrootconfigfile()
{
    if [ -n "$rootconfigfile" ]
    then
	configfile=$rootconfigfile
	return
    fi
    configfile=""
    while ( [ "$configfile" = "" ] || [ ! -f $configfile ] )
    do
        $echo "Server config filename:  ${nnl}" 
        read configfile
    done
    rootconfigfile=$configfile
}

# Install the root web
installrootweb()
{
    retval=0
    
    configfile=""
    admin=""
    port=""
    multihost=""
    webname=""
    webowner=""
    
    echo "  Note: Local version of Apache must use the FrontPage Apache patch."
    echo "  See http://www.microsoft.com/frontpage/ for more details."
    
    servertype="apache-fp"
    
    echo 
    echo "Note: If you have not installed the root web then you need to do it now." 
    echo
    myprompt 'yYnN' "Do you want to install a root web (y/n)" "Y"
    if [ $answer = n ] || [ $answer = N ]
    then
        return $retval
    fi
    
    echo " " 
    echo "Installing the root web..." 
    echo " " 
    
    webname="/"
    
    getrootconfigfile
    httpdconfigfile=$configfile
    
    admin=""
    until [ "$admin" != "" ]
    do
        $echo "FrontPage Administrator's user name:  ${nnl}" 
        read admin
    done
    httpdconfigfile=$configfile
    
    getHttpRootDirective $configfile Port
    port=$param
    
    until [ "$port" != "" ]
    do
        $echo "Enter the new server's port number:  ${nnl}" 
        read port
    done
    weconfigfile="${installdir}/we${port}.cnf"
    
    echo 
    getparam User $configfile $port "Getting User from "
    defwebowner=$param
    webowner=""
    until [ "$webowner" != "" ]
    do
        $echo "Unix user name of the owner of this new web: [$defwebowner]  ${nnl}"
        read webowner
        if [ "$webowner" = "" ]
        then
            webowner=$defwebowner
        fi
    done
    
    echo 
    getparam Group $configfile $port "Getting Group from "
    defgroup=$param
    webgroup=""
    until [ "$webgroup" != "" ]
    do
        $echo "Unix group of this new web: [$defgroup]  ${nnl}" 
        read webgroup
        if [ "$webgroup" = "" ]
        then
            webgroup=$defgroup
        fi
    done
    
    multihost="."
    if [ "$configfile" = "" ]
    then
        configfile="."
    fi
    
    PORT=$port
    
    echo "Installing root web into port $port..." 
    echo 
    installserver $port $multihost $webname $webowner $webgroup \
                  $admin $configfile $servertype ||
    {
        echo "ERROR:  $webname installation failed." 
        $echo "Hit enter to continue${nnl}" 
        read continue
        return $retval
    }
    
    return $retval
}

# set the global values for language and charactor encoding
handlelanguage()
{
    retval=0
    
    until [ "$charencoding" != "" ]
    do
        echo 
        echo "    1.  LATIN1 (ISO 8859-1)" 
        echo "    2.  LATIN2 (ISO 8859-2)" 
        echo "    3.  EUCJP  (Japanese EUC)" 
        echo "    4.  EUCKR  (Korean EUC)" 
        $echo "Which local character encoding does your system support: [1]  ${nnl}" 
        read charencodingnum
        echo
        
        case $charencodingnum in
           "") charencoding="latin1" ;;
          "1") charencoding="latin1" ;;
          "2") charencoding="latin2" ;;
          "3") charencoding="eucjp" ;;
          "4") charencoding="euckr" ;;
            *) echo "Invalid option!  Please try again."  ;;
        esac
    done
    
    until [ "$lang" != "" ]
    do
        echo 
        echo "    1.  English" 
        echo "    2.  French" 
        echo "    3.  German" 
        echo "    4.  Italian" 
        echo "    5.  Japanese" 
        echo "    6.  Spanish" 
        $echo "What should the default language be: [1] ${nnl}" 
        read langnum
        echo
        
        case $langnum in
            "") lang="en" ;;
           "1") lang="en" ;;
           "2") lang="fr" ;;
           "3") lang="de" ;;
           "4") lang="it" ;;
           "5") lang="ja" ;;
           "6") lang="es" ;;
             *) echo "Invalid option!  Please try again."  ;;
        esac
    done
    
    conffile=$FPDIR/frontpage.cnf
    echo "Setting "$conffile" to:" 
    echo 
    echo "defaultLanguage:${lang}"
    echo "localCharEncoding:${charencoding}"
    echo 
    echo "Copying ${conffile} to ${conffile}.orig"
    echo
    cp -p $conffile ${conffile}.orig ||
    {
        echo "ERROR:  Unable to backup $conffile to ${conffile}.orig!"
        $echo "Hit enter to continue${nnl}" 
        read continue
        return 1
    }
    
    echo "Creating and modifying new ${conffile}..."
    echo
    sed -e "s/defaultLanguage:.*/defaultLanguage:${lang}/g
            s/localCharEncoding:.*/localCharEncoding:${charencoding}/g" \
        ${conffile}.orig > $conffile ||
    {
        echo "ERROR:  Unable to create new $conffile!"
        echo "If the file has been corrupted you should be able to replace it with"
        echo "the backup file (${conffile}.orig)"
        $echo "Hit enter to continue${nnl}"
        read continue
        return 1
    }
    
    return $retval
}

# Install a web on a multihosted server
installvirtualwebs()
{
    retval=0
    
    echo
    echo "Installing Virtual Webs.." 
    echo 
    myprompt 'yYnN' "Do you want to install Virtual Webs (y/n)" "Y"
    if [ $answer = n ] || [ $answer = N ]
    then
        return $retval
    fi
    
    defaultconfigfile=$configfile
    configfile=""
    while ( [ "$configfile" = "" ] || [ ! -f $configfile ] )
    do
        $echo "Server config filename [${defaultconfigfile}]:  ${nnl}" 
        read configfile
        if [ "$configfile" = "" ]
        then
            configfile=$defaultconfigfile
        fi
    done
    httpdconfigfile=$configfile
    
    getHttpRootDirective $configfile Port
    port=$param
    getparam User $configfile $port "Getting User from "
    webowner=$param

    virtnames=`cat $configfile | $awk "
        BEGIN {
            value = \"\"; j=0
        }
        
        { x=0; y=0; server=\"\"; virtname=\"\" }
        
        /^[^#]* *< *[Vv][Ii][Rr][Tt][Uu][Aa][Ll][Hh][Oo][Ss][Tt][ $TAB]/ { 
            virtname = \\\$2; y=1; j++
        }
        
        /^[^#]*[Ss][Ee][Rr][Vv][Ee][Rr][Nn][Aa][Mm][Ee][ $TAB]/  { 
            server = \\\$2; y=2
        }
        
        {
            if (y == 1) {  gsub(/>/,\"\",virtname); ARRAY[j] = virtname }
            if (y == 2) { ARRAY[j] = server }
        }
        
        END { 
            for ( i =1; i <= j; i++ ) { 
                print ARRAY[i] \" \" 
            }
        }"`

    virtnames=`echo $virtnames | sed -e 's/>//g'`
    n=0
    for word in $virtnames 
    do
        n=`expr $n + 1` 
        eval virtwebs$n="$word" 
    done

    while echo
        n=1
        vweb="virtwebs$n"
        val=`eval echo '$'$vweb`
        while [ "$val" != "" ]
        do
            echo "  $n) $val" 
            n=`expr $n + 1`
            vweb="virtwebs$n"
            val=`eval echo '$'$vweb`
        done
        
        echo 
        $echo "Select the virtual web to install (CTRL-D if no more webs): ${nnl}" 
        read virtwebno
    do
        getHttpRootDirective $configfile Port
        port=$param
        until [ "$port" != "" ]
        do
            $echo "Enter the new server's port number:  ${nnl}" 
            read port
        done
        
        port="`eval echo '$'"virtwebs$virtwebno"`:$port"
        
        getHttpVirtualDirective $configfile $port User
        defwebowner=$param
        
        webowner=""
        until [ "$webowner" != "" ]
        do
            $echo "Unix user name of the owner of this new web:[$defwebowner]  ${nnl}"
            read webowner
            if [ "$webowner" = "" ]
            then
                webowner=$defwebowner
            fi
        done
        
        getHttpVirtualDirective $configfile $port Group
        defwebgroup=$param
        
        webgroup=""
        until [ "$webgroup" != "" ]
        do
            echo 
            $echo "Unix group of this new web:[$defgroup]  ${nnl}" 
            read webgroup
            if [ "$webgroup" = "" ]
            then
            webgroup=$defgroup
            fi
        done
        
        webname="/"
        
        admin=""
        until [ "$admin" != "" ]
        do
            $echo "FrontPage Administrator's user name:  ${nnl}" 
            read admin
        done
        
        servertype="apache-fp"
        
        multihost="`eval echo '$'"virtwebs$virtwebno"`"
        if [ "$configfile" = "" ]
        then
            configfile="."
        fi
        
        PORT=$port
        echo "Installing virtual root web into port $port..." 
        echo 
        installserver $port $multihost $webname $webowner $webgroup \
        $admin $configfile $servertype ||
        {
            echo "ERROR:  $webname installation failed." 
            $echo "Hit enter to continue${nnl}" 
            read continue
            return $retval
        }
        
        weconfigfile=""
        
        installnewsubwebs $PORT || error     # Install new servers
        echo
    done
    
    return $retval
}

# Install all subwebs, including per-user webs
installnewsubwebs()
{
    port=$1
    
    retval=0
    
    echo 
    myprompt 'yYnN' "Install new sub/per-user webs now (y/n)" "Y"
    if [ $answer = n ] || [ $answer = N ]
    then
        echo "For details on how to upgrade servers manually, please see" 
        echo "the Server Extension Resource Kit (SERK), located at"
        echo "http://www.microsoft.com/frontpage"
        echo
        return $retval
    fi
    echo 
    
    if [ "$weconfigfile" = "" ]
    then
        until [ "$weconfigfile" != "" ] && [ -f $weconfigfile ]
        do
            port=""
            if [ "$port" = "" ]
            then
                cat <<EOF
 
You need to specify which server to install the subweb into.
Examples:
          80
          www.virtual.web:80
 
          where 80 is the port the server is running on.
 
EOF
                $echo "Which server would you like to install the subweb into:  ${nnl}" 
                read port
            fi
            echo 
            case $port in 
              *:*) weconfigfile="${installdir}/${port}.cnf" ;;
                *) weconfigfile="${installdir}/we${port}.cnf" ;;
            esac
        done
    fi
    
    echo "Using FrontPage Configuration File:  ${weconfigfile}"
    echo
    
    servertype="apache-fp"
    configfile=`grep -i "^serverconfig:" $weconfigfile | sed -e '
                s/serverconfig://g
                s/fakeconf.*\///'`
    configfiledir=`dirname $configfile`"/"
    
    getdocroot $weconfigfile ||
    {
        echo "ERROR:  Unable to get DocumentRoot/UserDir"
        return 1
    }
    
    services=${docroot}"/_vti_pvt/services.cnf"
    
    if [ ! -f $services ]
    then
        echo
        echo "ERROR:  The root web must be installed before sub/per-user webs!"
        echo "Document Root:  ${docroot}"
        $echo "Hit enter to continue${nnl}" 
        read continue
        echo
        return $retval
    fi
    
    echo 
    while $echo "Enter the web name (CTRL-D if no more webs): ${nnl}" 
        read webname    
    do
        admin=""
        until [ "$admin" != "" ]
        do
            $echo "FrontPage Administrator's user name:  ${nnl}" 
            read admin
        done
        
        case "$webname" in
            ~*) PERUSER=1
                echo 
                echo "Web $webname is a per-user web" 
                webowner=`echo $webname | sed -e 's/~//'`
                ;;
             *) PERUSER=0
                echo 
                echo "Web $webname is a subweb"  
                webowner=""
                ;;
        esac
        
        defwebowner=`$lsg ${docroot}${service} | $awk ' { print $3}'`
        until [ "$webowner" != "" ]
        do
            echo
            $echo "Unix user name of the owner of this new web:[$defwebowner]  ${nnl}"
            read webowner
            if [ "$webowner" = "" ]
            then
                webowner=$defwebowner
            fi
        done

        getparam Group $configfile $port "Getting Group from "
        defgroup=$param
        webgroup=""
        if [ "$PERUSER" = 1 ]
        then
            getHomedir $webowner
            if [ -d "${homedir}" ]
            then
                webgroup=`$lsg ${homedir} | $awk ' { print $4}'`
            fi
            if [ -d "${homedir}/${userdir}" ]
            then
                webgroup=`$lsg ${homedir}/${userdir} | $awk ' { print $4}'`
            fi
        fi
        
        until [ "$webgroup" != "" ]
        do
            echo 
            $echo "Unix group of this new web:[$defgroup]  ${nnl}" 
            read webgroup
            if [ "$webgroup" = "" ]
            then
                webgroup=$defgroup
            fi
        done
        
        multihost="."
        
        if [ "$configfile" = "" ]
        then
            configfile="."
        fi
        
        if [ "$servertype" = "" ]
        then
            servertype="."
        fi
        
        installserver $port $multihost $webname $webowner $webgroup \
                      $admin $configfile $servertype ||
        {
            echo "ERROR:  $webname installation failed." 
            $echo "Hit enter to continue${nnl}"
            retval=0 
            read continue
        }
        echo
    done
    
    return $retval
}

installserver()
{
    retval=0
    
    webport="$1"
    multihost="$2"
    webname="$3"
    webowner="$4"
    webgroup="$5"
    admin="$6"
    configfile="$7"
    servertype="$8"

    case $webport in
       *:*) multihost=`echo $webport | cut -d: -f1`
            port=`echo $webport | cut -d: -f2`
            ;;
         *) port=$webport
            ;;
    esac
    
    echo 
    echo "installing server " $webname " on port " $port
    
    if [ "$PERUSER" = 1 ]
    then
        user=`echo $webowner | sed -e 's/~//g'`
        echo 
        echo "Will chown per-user web to $user as part of install." 
        echo "Will chgrp per-user web to $webgroup as part of install." 
        chown="-xUser $user -xGroup $webgroup"  
    else
        if [ "$webowner" != "." ]
        then
            echo 
            echo "Will chown web to $webowner as part of install." 
            echo "Will chgrp web to $webgroup as part of install." 
            chown="-xUser $webowner -xGroup $webgroup"
        else
            chown=""
        fi
    fi
    
    if [ "$configfile" != "." ]
    then
        config="-s $configfile"
    else
        config=""
    fi
    
    if [ "$webname" != "/" ]
    then 
        web="-w $webname"
    else
        web=""
    fi

    if [ "$multihost" != "." ]
    then
        ${FPDIR}/bin/owsadm.exe -o install -p $port $web $config -u $admin $chown -m $multihost -t $fpversionoption || retval=1
    else
        ${FPDIR}/bin/owsadm.exe -o install -p $port $web $config -u $admin $chown -m "" -t $fpversionoption || retval=1
    fi
    
    return $retval
}

###############################################################################
#
# The following set of functions are used for reading the webserver's config
# file(s) (i.e. httpd.conf)
#

# take a string and converts it to a regular expression which matches all
# possible case combinations.
#
# String = [Ss][Tt][Rr][Ii][Nn][Gg]
#
getMixedCase() {
    mc_string=`echo $1 | $awk "
                BEGIN {
                    newstring = \"\";
                    oldstring = \"$1\"; 
                } 
                
                $functions
                {
                    strlen = length(oldstring);
                    for ( i = 1; i <= strlen; ++i ) {
                        char = substr(oldstring, i, 1)
                        newstring = newstring \"[\" toupper(char) tolower(char) \"]\";
                    };
                    print newstring
                }"`
}

getDirective()
{
    gdconfigfile=$1
    directive=$2
    port=$3

    param=""
    if [ "$gdconfigfile" = "" ]
    then
        return 0
    fi
    
    case $port in
      *:*) getHttpVirtualDirective $gdconfigfile $port $directive ;;
        *) getHttpRootDirective $gdconfigfile $directive ;;
    esac
}

# Get the DocRoot and UserDir settings
getdocroot()
{
    weconfigfile=$1
    
    servertype="apache-fp"
    configfile=`grep -i "^serverconfig:" $weconfigfile | sed -e '
                s/serverconfig://g
                s/fakeconf.*\///'`
    
    echo "Getting DocumentRoot and UserDir."
    
    if [ ! -f "$configfile" ]
    then
        echo "$configfile does not exist."
        return 1
    fi
    
    configfiledir=`dirname $configfile`"/"
    
    docroot=""
    resconffile=$configfile
    echo "Getting resource configuration file"
    getHttpDirective $configfile ResourceConfig $port
    if [ "$param" != "" ]
    then
        case $param in
            /dev/null) resconffile=$configfile
                ;;
            /*) resconffile=$param  # fully qualified pathname
                ;;
            *)  file=`basename $param`
                resconffile="${configfiledir}${file}"
                ;;
        esac
    elif [ $httpdmajor -eq 1 ]
    then
    # only Apache 1 requires srm.conf
        resconffile="${configfiledir}srm.conf"
        if [ ! -f $resconffile ]
        then
            resconffile=$configfile
        fi
    else
       resconffile=$configfile
    fi
    
    if [ $httpdmajor -eq 1 -a "$resconffile" != "" -a "$resconffile" != "/dev/null" -a ! -f "$resconffile" ]
    then
        echo "ERROR:  $resconffile does not exist!"
        return 1
    fi
        
    getparam DocumentRoot $resconffile $port "Getting DocumentRoot from "
    docroot=$param

    getparam UserDir $resconffile $port "Getting UserDir from "
    userdir=$param

    if [ "${docroot}" = "" ]
    then
        echo "ERROR:  DocumentRoot not defined!"
        return 1
    fi
    
    if [ ! -d "${docroot}" ]
    then
        echo "ERROR:  ${docroot} does not exist!"
        return 1
    fi
    
    echo
    echo "DocumentRoot: ${docroot}"
    
    if [ "${userdir}" = "" ]
    then
        echo "WARNING:  UserDir not defined."
    else
        echo "UserDir: ${userdir}"
    fi
    
    echo
    
    return 0
}

# call either getHttpVirtualDirective or getHttpRootDirective
getHttpDirective()
{
    configfile=$1
    directive=$2
    port=$3
    
    getDirective "$configfile" "$directive" "$port"
    
    if [ "$param" = "" ]
    then
        echo "Directive $directive not found."
    else 
        echo "Found Directive $directive, value $param."
    fi
}

# find a directive in a config file that is not part of a virtual host section
getHttpRootDirective()
{
    configfile=$1
    directive=$2
    
    getMixedCase VirtualHost
    virtualhost=${mc_string}
    
    getMixedCase $directive
    
    param=`cat $configfile | $awk "
            /^[^#]* *< *${virtualhost}/,/^[^#]* *< *\/${virtualhost}/ { next }
            /^[^#]* *${mc_string}[ $TAB]/  { print \\\$2 }" | sed -e 's/^\"//' -e 's/\"$//' ` 
            
    return 0
}

# find a directive in a config file that is part of a virtual host section
getHttpVirtualDirective() # from fp_install.sh
{
    configfile=$1
    port=$2
    directive=$3
    
    virtweb=`echo  $port | sed -e 's/:[0-9]*.$//'`
    virtweb=`basename $virtweb`
    
    getMixedCase VirtualHost
    virtualhost=${mc_string}
    
    getMixedCase ServerName
    servername=${mc_string}
    
    getMixedCase $directive
    
    param=`cat $configfile | $awk "
                            BEGIN    { value = \"\"; j=0 }
                                     { x=0; y=0; server=\"\"; virtname=\"\" }
                        /^[^#]* *< *${virtualhost}[ $TAB]/ { 
                            virtname = \\\$2; y=1; j++
                        }
                        /^[^#]*${servername}[ $TAB]/   { 
                            server = \\\$2; y=2
                        }
                        /^[^#]* *${mc_string}[ $TAB]/  {
                            arg = \\\$2; y=3
                        }
                        
                        {
                            if (y == 1) {
                                gsub(/>/,\"\",virtname)
                                VIRTNAMES[j] = virtname
                            }
                            if (y == 2) { SERVERNAMES[j] = server } 
                            if (y == 3) { ARRAY[j] = arg }
                        }
                    
                    END { 
                            for ( i =1; i <= j; i++ ) { 
                                if (SERVERNAMES[i] == \"$virtweb\" || VIRTNAMES[i] == \"$virtweb\" ) {
                                    print ARRAY[i] 
                                }
                            }
                    } "`
    
    if [ "$param" = "" ]
    then
        getHttpRootDirective $configfile $directive
    fi 
}

# get the value of parameters from the config file
getparam() {
    directive=$1
    gpconfigfile=$2
    port=$3
    msg=$4

    getDirective "$httpdconfigfile" "$directive" "$port"   # try httpd.conf
    if [ "$param" = "" -a -f "$gpconfigfile" ]       # try again w/configfile
    then
        getDirective "$gpconfigfile" "$directive" "$port"
    else
        gpconfigfile=$httpdconfigfile
    fi
    echo $msg $gpconfigfile
    if [ "$param" = "" ]
    then
        return 1;
    else
        return 0;
    fi
    param=`egrep -i "^[ $TAB]*$1[ $TAB]" $2 | $awk '{print $2}' | sed -e 's/^\"//' -e 's/\"$//'` || return 1
    return 0;
}

###############################################################################
# (formerly a separate script, change_server.sh)
# these functions are used to replace the existing Apache daemon (httpd) with a
# new one that contains the FrontPage patch

change_server()
{
    rm -f ${FPDIR}/apache-fp/suidkey
    
    # Generate new key file
    ps=`$psea | tail -10` ; echo $ps | sed -e 's/ //g' | cut -c10-137 > ${FPDIR}/apache-fp/suidkey
    
    chown root ${FPDIR}/apache-fp/suidkey
    chmod 600 ${FPDIR}/apache-fp/suidkey

    
    checkVersion
    rtc=$?
    if [ $rtc -eq 1 ]
    then
        # User doesn't want us to upgrade Apache, so we abort.
        exit 0
    elif [ $rtc -eq 2 ]
    then 
        # Already upgraded, so we return
        httpdmajor=1
        fpversionoption="apache-fp"
        return 0
    elif [ $rtc -eq 3 ]
    then
        # Apache 2.0
	httpdmajor=2
        fpversionoption="apache-2.0"
	installfpdso2 || 
	{
	    echo "ERROR: Unable to install mod_frontpage dso"
	    return 1
	}
	return 0
    elif [ $rtc -eq 4 ]
    then
	# Apache 1.3 with mod_so
	httpdmajor=1
	fpversionoption="apache-fp"
	installfpdso1 || 
	{
	    echo "ERROR: Unable to install mod_frontpage dso"
	    return 1
	}
	return 0
    elif [ $rtc -ne 0 ]
    then
        # Unknown return value from checkVersion
        echo "ERROR: checkVersion returned $rtc, aborting."
	exit 1
    fi
    
    not_stub_apache         || error   # Check if we're debugging this script
    if [ $rtc -eq 0 ]
    then
        upgradeHttpd || 
	{
	    echo "ERROR:  Unable to upgrade Apache daemon!"
	    return 1
	}
    fi
    
    httpdmajor=1
    fpversionoption="apache-fp"
    return 0
}

# checkVersion:
# Use the httpd -v command to get the built-in Apache version number.  If it
# is a Version 1.3 apache, then run "strings" on the httpd file to see if 
# the current version of the FP patch is installed; ask the user if they 
# want to upgrade to the current one.
# Return values:
#    0:  Apache is version 1 and needs to be upgraded by us
#    1:  Apache is old but User does not want to upgrade Apache (for an old
#        Apache 1), or else it's an old Apache 2, which users must upgrade
#        themselves.  This will cause change_server (the caller of this
#        function) to abort with no message (thus, the messages must be
#        issued here).
#    2:  Apache is version 1, but it has the FP patch
#    3:  Apache is version 2
#    4:  Apache is correct version 1, with mod_so and no FrontPage, so we
#        install the mod_frontpage.so (not the patched version).

checkVersion()
{
    rtc=0;
    echo
    NEWHTTPD=$NEWHTTPDNEW
    
    # First check for Apache 2
    if [ "$ApacheVers" = "2.0"  ]
    then
      # It's Apache 2.0, check the details, and if ok, return.  We don't
      # ship a prebuilt Apache 2.0, nor is there a patch to install
      checkVersion2 $installedApacheVers
      rtc=$?
      unset installedApacheVers
      return $rtc
    fi

    if [ 1 -ne `expr "$installedApacheVers" : "1"` ]
    then
      # Unknown
      die1 "Unknown Apache version, aborting"
    fi

    checkVersion1 $installedApacheVers
    rtc=$?
    unset installedApacheVers
    return $rtc
}

#
# checkVersion1 ApacheCompare
#
#     ApacheCompare must be one of "LT", "EQ", or "GT", representing the
#     comparison of the version in this kit agains the version the user
#     has installed
#
# Make sure we have the correct version of Apache 1.3.
# If it's too old, we recommend mod_so, but will continue with our version
# 
checkVersion1()
{
    # First collect the version numbers for the Apache in our kit, and
    # compare them
    sourceVersion=`$NEWHTTPD -v | sed 's%^.*Apache/\([0-9\.]*\).*%\1%' | head -1`
    sourceFPVersion=`$strings $NEWHTTPD | $fgrep "FrontPage/" 2> /dev/null | sed -e 's%^.*FrontPage/\([0-9\.]*\).*%\1%' | tail -1`
    compareVersions "$sourceVersion" "$installedApacheVers" ApacheCompare

    # Now get the FrontPage patch versions, and compare them.
    echo "Checking if the FrontPage patch to Apache is already installed."
    targetFPVersion=`$strings $HttpdFile | $fgrep "FrontPage/" 2> /dev/null | sed -e 's%^.*FrontPage/\([0-9\.]*\).*%\1%' | tail -1`
    if [ "${targetFPVersion}" = "" ]
    then
	# We treat the case of no patch the same as if it is an older patch,
        # except for the following message.  Hence setting FPCompare to GT
        # works.
        echo "The FrontPage server patch has not been installed in your server"
        FPCompare="NO"
    else
	echo "Your server has the FrontPage patch ${targetFPVersion} installed."
        compareVersions "$sourceFPVersion" "$targetFPVersion" FPCompare
    fi

    # All the comparisons are done, dispatch the recommendations.
    badCompare=""
    if [ "$ApacheCompare" = "GT" ]
    then
      case "$FPCompare" in
        NO)  recommendation1gt_no ; rtc=$?;;
	GT)  recommendation1gt_gt ; rtc=$?;;
	EQ)  recommendation1gt_eq ; rtc=$?;;
	LT)  recommendation1gt_lt ; rtc=$?;;
	*)   badCompare="FP=$FPCompare"  ; rtc=1;;
      esac
    elif [ "$ApacheCompare" = "EQ" ]
    then
      case "$FPCompare" in
        NO)  recommendation1eq_no ; rtc=$?;;
	GT)  recommendation1eq_gt ; rtc=$?;;
	EQ)  recommendation1eq_eq ; rtc=$?;;
	LT)  recommendation1eq_lt ; rtc=$?;;
	*)   badCompare="FP=$FPCompare"  ; rtc=1;;
      esac
    elif [ "$ApacheCompare" = "LT" ]
    then
      case "$FPCompare" in
        NO)  recommendation1lt_no ; rtc=$?;;
	GT)  recommendation1lt_gt ; rtc=$?;;
	EQ)  recommendation1lt_eq ; rtc=$?;;
	LT)  recommendation1lt_lt ; rtc=$?;;
	*)   badCompare="FP=$FPCompare"  ; rtc=1;;
      esac
    else
      badCompare="AP=$ApacheCompare"
      rtc=1
    fi

    if [ "$badCompare" != "" ] ;
    then
      echo "Internal installation error, bad value is $badCompare"
      exit 1
    fi
    return $rtc
}


recommend_upgrade()
{
    echo "We recommend installing Apache $sourceVersion with mod_so."
    echo "Alternatively, we can install a prebuilt Apache $sourceVersion "
    echo "that has been patched to work with this version of FrontPage."
}

warnLoseCustomModules()
{
    echo "Our prebuilt Apache is a stock Apache containing the FrontPage "
    echo "patch and may result in the loss of any nonstandard modules "
    echo "that have been compiled into the Apache daemon that you have"
    echo "currently installed."
}

ask_upgrade()
{
    recommend_upgrade
    warnLoseCustomModules
    myprompt 'yYnN' "Are you sure you want to install the prebuilt Apache? (y/n)" "N"
    if [ $answer = n ] || [ $answer = N ]
    then
      echo 
      echo "Please install Apache $sourceVersion with mod_so, then rerun"
      echo "this installation."
      return 1
    else
      return 0
    fi
}

recommendation1gt_no()
{
    # We have a newer Apache, and they have no FP patch
    echo "Your Apache needs to be upgraded."
    ask_upgrade
    rtc=$?
    return $rtc
}

recommendation1gt_gt() 
{
    # We're newer on both counts
    echo "Your Apache needs to be upgraded, along with the FrontPage patch."
    ask_upgrade
    rtc=$?
    return $rtc
}

recommendation1gt_eq()
{
    # We have a newer Apache, but they have the same FP patch
    echo "Your Apache needs to be upgraded."
    ask_upgrade
    rtc=$?
    return $rtc
}

recommendation1gt_lt()
{
    # We have a newer Apache, but an older FP patch.
    echo "You are currently running an older Apache with a more recent "
    echo "FrontPage patch.  That configuration is not recommended."
    echo "Installing this kit requires a newer Apache, but will result in"
    echo "a downgrade of your FrontPage patch."
    myprompt 'yYnN' "Do you want to continue? (y/n)" "N"
    if [ $answer = n ] || [ $answer = N ]
    then
      return 1
    fi
    ask_upgrade
    rtc=$?
    return $rtc
}

getModFpVersion1() 
{
    httpdir=`dirname $HttpdFile`
    # Remove the /bin
    httpdir=`dirname $httpdir`/libexec
    if [ ! -d "$httpdir" ]
    then
	httpdir="/usr/local/apache/modules"
    fi
    while ( [ ! -d "$httpdir" ] )
    do 
	$echo "Where is the current Apache modules directory: [$httpdir] ${nnl}"
	read location
	if [ "$location" != "" ]
	then
	    httpdir=$location
	fi
    done
    modfpso="${httpdir}/mod_frontpage.so"
}

checkModFpSo1()
{
    getModFpVersion1
    if [ ! -r "$modfpso" ]
    then
      # No mod_frontpage.so
      echo "The mod_frontpage.so module is not installed, but mod_so is."
      echo "You can let us install mod_frontpage.so, or we can install"
      echo "a prebuilt Apache with mod_frontpage built in"
      myprompt 'yYnN' "Do you want us to install the mod_frontpage.so DSO? (y/n)" "Y"
      if [ $answer = y ] || [ $answer = Y ]
      then
          return 4
      else
          return 0
      fi
    else
      # There is a mod_frontpage.so, check version
      targetModFPVersion=`$strings $modfpso | $fgrep "FrontPage/" 2> /dev/null | sed -e 's%^.*FrontPage/\([0-9\.]*\).*%\1%' | tail -1`
      sourceModFPVersion=`$fgrep "FrontPage/" "$NEWMODFPSO1" 2> /dev/null | sed -e 's%^.*FrontPage/\([0-9\.]*\).*%\1%' | tail -1`
      compareVersions "$sourceModFPVersion" "$targetModFPVersion" ModFPCompare
      case "$ModFPCompare" in
        GT) echo "Your existing mod_frontpage.so will be upgraded"
	    return 4;;
	EQ) echo "You are already running mod_frontpage.so, no changes will be made"
	    return 2;;
	LT) echo "You are running mod_frontpage.so version $targetModFPVersion"
	    echo "Installing this version will be a downgrade."
            myprompt 'yYnN' "Are you sure that you want to do this (y/n)" "N"
            if [ $answer = n ] || [ $answer = N ]
            then
                return 1
	    else
	        return 4
            fi;;
	 *) echo "Internal installation error, bad value is ModFPCompare=$ModFPCompare"
	    exit 1;;
      esac
  fi
}


recommendation1eq_no()
{
    # Right Apache, no FP.
    # Check for mod_so
    hasSo=`$HttpdFile -l | grep " mod_so.c"`
    if [ "$hasSo" = "" ]
    then
	echo "Your Apache is the correct version, but does not have mod_so."
	echo "We recommend rebuilding Apache with mod_so.  Alternatively,"
	echo "we can install a prebuilt Apache that has been patched to"
	echo "work with FrontPage."
	warnLoseCustomModules
	myprompt 'yYnN' "Do you really want to install the prebuilt Apache? (y/n)" "N"
	if [ $answer = n ] || [ $answer = N ]
	then
	    echo 
	    echo "Please build Apache $sourceVersion with mod_so, then rerun"
	    echo "this installation."
	    return 1
	else
	    return 0
	fi
    fi

    # At this point, it has mod_so, so we continue
    checkModFpSo1
    rtc=$?
    if [ "$rtc" -eq 4 ]
    then
      echo "You will need to restart Apache after this installation completes."
    fi
    return $rtc
}

recommendation1eq_gt()
{
    echo "Your Apache is the correct version, but it has an older FrontPage"
    echo "patch that must be upgraded."
    echo "You can rebuild Apache with mod_so and without the existing"
    echo "FrontPage patch, and then rerun this installation to install"
    echo "the FrontPage module using Apache's mod_so.  Alternatively, "
    echo "we can install a prebuilt Apache of the same version, with the"
    echo "current FrontPage patch."
    warnLoseCustomModules
    myprompt 'yYnN' "Do you want to install the prebuilt Apache? (y/n)" "N"
    if [ $answer = n ] || [ $answer = N ]
    then
      echo 
      echo "Please install Apache $sourceVersion with mod_so, then rerun"
      echo "this installation."
      return 1
    else
      return 0
    fi
}

recommendation1eq_eq()
{
    echo "Your Apache server is currently the correct version, with the "
    echo "correct patch."
    return 2
}

recommendation1eq_lt()
{
    echo "Your Apache is the correct version, but it has a newer FrontPage"
    echo "patch.  That configuration is not recommended."
    echo "Installing this kit requires a downgrade of your FrontPage patch"
    myprompt 'yYnN' "Do you want to continue? (y/n)" "N"
    if [ $answer = n ] || [ $answer = N ]
    then
      return 1
    fi
    ask_upgrade
    rtc=$?
    return $rtc
}

warnNewApache() {
    echo "You are running a newer version of Apache than supported by"
    echo "this kit.  Your Apache should work with this version, but check"
    echo "our release notes to be sure."
}
recommendation1lt_no()
{
    hasSo=`$HttpdFile -l | grep " mod_so.c"`
    if [ "$hasSo" = "" ]
    then
        warnNewApache
	echo "Your Apache does not have mod_so."
	echo "Your choices are to rebuild Apache with mod_so, and then"
	echo "rerun this installation, or to have us install our prebuilt"
	echo "version."
	warnLoseCustomModules
	myprompt 'yYnN' "Do you want to install the prebuilt Apache? (y/n)" "N"
	if [ $answer = n ] || [ $answer = N ]
	then
	    echo 
	    echo "Please install Apache with mod_so, then rerun"
	    echo "this installation."
	    return 1
	else
	    return 0
	fi
    fi
    #At this point, we know they have mod_so installed.
    warnNewApache
    getModFpVersion1
    if [ ! -r "$modfpso" ]
    then
      # At this point, we know the user has a newer Apache with mod_so but
      # without mod_frontpage.
      echo "The mod_frontpage.so modules is not installed."
      echo "Your choices are:"
      echo "   Let us install our mod_frontpage.so into your existing Apache"
      echo "   Install Apache $sourceVersion with mod_so."
      echo "   Install Apache $sourceVersion with our patch." 
      myprompt 'yYnN' "Do you want us to install mod_frontpage.so into your existing Apache (y/n)" "N"
      if [ $answer = y ] || [ $answer = Y ]
      then
	return 4
      fi
      echo 
      echo "Please install the apropriate Apache, and then rerun"
      echo "this installation."
      return 1
    else
      # At this point, we know the user has a newer Apache with mod_so
      # and with mod_frontpage installed, but we don't know the version
      checkModFpSo1
      rtc=$?
      if [ "$rtc" -eq 4 ]
      then
        echo "You will need to restart Apache after this installation completes."
      fi
      return $rtc
    fi
}

recommendation1lt_gt()
{
    # Newer Apache, older patch (not mod_so)
    echo "You are currently running a newer Apache with an older FrontPage patch"
    echo "Your choices are to rebuild your Apache with mod_so and without the"
    echo "FrontPage patch, and then rerun this installation, or we can "
    echo "install a pre-built, older version of Apache with the current"
    echo "FrontPage patch built-in.  This would be an Apache downgrade."
    warnLoseCustomModules
    myprompt 'yYnN' "Do you want to downgrade to our prebuilt Apache? (y/n)" "N"
    if [ $answer = n ] || [ $answer = N ]
    then
        echo 
	echo "Please install the apropriate Apache, and then rerun"
	echo "this installation."
	return 1
    fi
    return 0
}

recommendation1lt_eq()
{
    # Newer Apache, current patch
    echo "You are currently running a newer version of Apache, but with the"
    echo "correct FrontPage patch.  We will use this version"
    return 2
}

recommendation1lt_lt()
{
    # Newer Apache, newer FrontPage
    echo "You are currently running a newer version of Apache ($installedApacheVers)"
    echo "with a newer version of FrontPage ($targetFPVersion).  Please check "
    echo "the release notes to ensure this configuration is supported."
    echo "We will continue the installation without changing your Apache."
    return 2
}
    

recommendation1()
{ 
    echo "This version of FrontPage requires either:"
    echo "  Apache Version $sourceVersion"
    echo "or"
    echo "  Apache Version APACHE_2_0_VERSION (with mod_so)"
    echo 
    echo "We recommend you install one of these versions, with mod_so."
    echo "Alternatively, we can install a prebuilt Apache $sourceVersion ."
    echo "This prebuilt version does not use mod_so for FronPage, but instead"
    echo "has been patched to work with FrontPage."
    myprompt 'yYnN' "Do you want to install this prebuilt version (y/n)" "N"
    if [ $answer = n ] || [ $answer = N ]
    then
      return 1
    fi
    return 0  
}    

#
# checkVersion2  targetApacheVersion
#
# Make sure we have the correct version of Apache 2.  We refuse to
# install if it's too old, and we require confirmation if it's too new.
#
# Make sure the required modules are compiled in.
# Currently, we abort if they're not, and return the result code for Version
# 2 (which happens to be 3, because 2 was already in use; see checkVersion
# for the values).
#
checkVersion2()
{
    installedApacheVers=$1
    ApacheCompare=""
    compareVersions "$APACHE_2_0_VERSION" "$installedApacheVers" ApacheCompare
    if [ "$ApacheCompare" != "EQ" -a "$ApacheCompare" != "LT" ]
    then
      echo "For Apache 2.0, FrontPage requires Apache version $sourceVersion or later."
      die1 "Please first upgrade to Apache 2.0.40 or later, with mod_so."
    fi

    if [ "$ApacheCompare" = "LT" ]
    then
      echo "WARNING: The Apache you are running is newer than the one"
      echo "supported by this kit."
      echo "You are running Apache $installedApacheVers ."
      echo "This kit supports Apache $sourceVersion ."
      myprompt 'yYnN' "Do you wish to continue? (y/n)" "N"
      if [ $answer = n ] || [ $answer = N ]
      then
          exit 0
      fi
    fi

    hasSo=`$HttpdFile -l start | grep " mod_so.c"`
    if [ "$hasSo" = "" ]
    then
	echo "Your Apache is missing one or more modules: ${nnl}"
	if [ "$hasSo" = "" ] ;     then echo "mod_so ${nnl}" ; fi
	echo "; these modules are required."
	die1 "Please rebuild Apache with all requied modules"
    fi
    # Set the return 
    rtc=3 
    return $rtc
}

# Replace the old httpd daemon with the new httpd daemon (after making backup)
upgradeHttpd()
{
    echo "Upgrading FrontPage Webs..."
    cp -p $HttpdFile ${HttpdFile}.orig || 
    {
        echo
        echo "ERROR:  Unable to copy ${HttpdFile} to ${HttpdFile}.orig" 
        return 1
    }
    
    cp ${NEWHTTPD} ${HttpdFile} || 
    {
        echo
        echo "ERROR:  Unable to copy ${NEWHTTPD} to ${HttpdFile}"
        return 1
    }
    
    return 0
}


installfpdso1()
{
    httpdir=`dirname $HttpdFile`
    while ( [ ! -d "$httpdir" ] )
    do 
	httpdir="/usr/local/apache/bin"
	$echo "Where is the current Apache 1.3 bin directory: [$httpdir] ${nnl}"
	read location
	if [ "$location" != "" ]
	then
	    httpdir=$location
	fi
    done
    if [ ! -x "$httpdir/apxs" ]
    then
	echo "Cannot find Apache apxs at $httpdir/apxs"
	return 1
    fi
    echo "Using the Apache apxs utility to build mod_frontpage.so"
    CURPATH=$PATH
    PATH=$OLDPATH
    case "$machine" in 
       freebsd) opts=-Wc,-DFREEBSD ;;
       bsdi)    opts=-Wc,-Dbsdi ;;
       linux)   opts=-Wc,-Dlinux ;;
       *)       opts= ;;
    esac
    $httpdir/apxs -c $opts $FPDIR/apache-fp/mod_frontpage.c
    res=$?
    PATH=$CURPATH
    if [ $res -ne 0 ]
    then
	return $res
    fi
    
    echo "Using the Apache apxs utility to install mod_frontpage.so"
    PATH=$OLDPATH
    $httpdir/apxs -i -a -n frontpage $FPDIR/apache-fp/mod_frontpage.so
    res=$?
    PATH=$CURPATH
    echo "Finished running the Apache apxs utility."
    return $res

}

installfpdso2()
{
    httpdir=`dirname $HttpdFile`
    while ( [ ! -d "$httpdir" ] )
    do 
	httpdir="/usr/local/apache/bin"
	$echo "Where is the current Apache 2.0 bin directory: [$httpdir] ${nnl}"
	read location
	if [ "$location" != "" ]
	then
	    httpdir=$location
	fi
    done
    if [ ! -x "$httpdir/apxs" ]
    then
	echo "Cannot find Apache apxs at $httpdir/apxs"
	return 1
    fi
    echo "Using the Apache apxs utility to build mod_frontpage.so"
    CURPATH=$PATH
    PATH=$OLDPATH
    case "$machine" in 
       freebsd) opts=-Wc,-DFREEBSD ;;
       bsdi)    opts=-Wc,-Dbsdi ;;
       linux)   opts=-Wc,-Dlinux ;;
       *)       opts= ;;
    esac
    $httpdir/apxs -c $opts $FPDIR/apache2/mod_frontpage.c $FPDIR/apache2/mod_fpcgid.c
    res=$?
    PATH=$CURPATH
    if [ $res -ne 0 ]
    then
	return $res
    fi
    echo "Using the Apache apxs utility to install mod_frontpage.so"
    PATH=$OLDPATH
    $httpdir/apxs -i -a -n frontpage $FPDIR/apache2/mod_frontpage.la
    res=$?
    PATH=$CURPATH
    echo "Finished running the Apache apxs utility."
    return $res

}

main
