/* 
   Copyright (C) 2000 David Rysdam

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License
   as published by the Free Software Foundation; either version 2
   of the License, or (at your option) any later version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
   
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

   See COPYING for details.
*/

#include <stdio.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netdb.h>
#include <netinet/in.h>
#include <fcntl.h>

/*
  Algorithm:
  
  1) Email to send will be received on stdin
  2) Open defined host and port number
  3) Send mail out port with SMTP protocol as:
     a) HELO defined domain
     b) MAIL FROM: defined sender (from MUA?)
     c) RCPT TO: to header from MUA
     d) DATA
     e) mail file
     d) <CRLF>.<CRLF>

*/

/*
  get options
  call buffer_mail
  call parse_mail
  call send_mail
*/

#define BUF_SIZE 64
#define MAX_MSG_LEN 1024*5
#define MAX_RCPTS 8
#define MAX_RCPT_ADDR_LEN 128

char *host=NULL;
char *fromaddr=NULL;
char *domain=NULL;
int port=0;
int debug_level=0;
int num_rcpts=0;
FILE *stdlog;

char buffer[MAX_MSG_LEN+1];
//yes, I realize this is ugly AND stupid--but do you have a better idea?
char crlf_buffer[MAX_MSG_LEN*2+1];
char rcpts[MAX_RCPTS][MAX_RCPT_ADDR_LEN];
int more_data=1;

int buffer_mail(void);
int parse_mail(void);
int send_mail(void);
int get_socket(void);

void print_usage(char *prog)
{
  printf("Usage:\n");
  printf("%s -d domain -f from@addr -h host [-p port] [-l debuglevel]\n", prog);
}

int main(int argc, char *argv[])
{
  int i;

  for(i=1; i<argc; i+=2){
    switch(*(argv[i]+1))
      {
      case 'h':
	host = (char *)strdup(argv[i+1]);
	break;
      case 'd':
	domain = (char *)strdup(argv[i+1]);
	break;
      case 'f':
	fromaddr = (char *)strdup(argv[i+1]);
	break;
      case 'p':
	port = atoi(argv[i+1]);
	break;
      case 'l':
	debug_level =  atoi(argv[i+1]);
	if(debug_level > 1)
	  stdlog = fopen("nbsmtp.log", "w");
	else
	  stdlog = stdout;
	break;
      default:
	print_usage(argv[0]);
      }
  }

  if(domain==NULL || fromaddr==NULL || host==NULL){
    print_usage(argv[0]);
    return 1;
  }

  if(port==0)
    port = 25;

  if(debug_level){
    fprintf(stdlog, "Running with options:\n");
    if(host)
      fprintf(stdlog, "\thost = %s\n", host);
    fprintf(stdlog, "\tport = %d\n", port);
    fprintf(stdlog, "\tdebuglevel = %d\n", debug_level);
  }

  if(buffer_mail() > 0){
    fprintf(stdlog, "Error in buffer_mail\n");
    return 1;
  }

  if(parse_mail()){
    fprintf(stdlog, "Error in parse_mail\n");
    return 1;
  }

  if(send_mail()){
    fprintf(stdlog, "Error in send_mail\n");
    return 1;
  }

  return 0;
}

/*
  allocate space for buffer
  read stdin into buffer
*/
int buffer_mail(void)
{
  if(!more_data)
    return 0;

  memset(buffer, 0, MAX_MSG_LEN+1);
  
  if(fread(buffer, 1, MAX_MSG_LEN, stdin) == 0)
    return 1;
  
  if(debug_level)
    fprintf(stdlog, "Buffer:\n%s\n", buffer);
  
  more_data = (feof(stdin)?0:-1); 
  if(debug_level && !more_data)
    fprintf(stdlog, "EOF\n", buffer);

  return more_data;
}

/*
  find To: line (search for "\nTo: ")
  find CC: line (search for "\nCC: ")
  find BCC: line (search for "\nBCC: ")
  create array of rcpts from comma or semi-colon-sep list
*/
int parse_mail(void)
{
  char *toline, *endheader;
  char *headers[7] = {"To: ", "CC: ", "Cc: ", "cc: ", "BCC: ", "Bcc: ", "bcc: "};
  int cur_rcpt=0, cur_pos=0, cur_header;
  
  endheader = (char *)strstr(buffer, "\n\n");
  if(endheader == NULL)
    return 1;
  
  for(cur_header=0; cur_header<7; cur_header++){
    toline = (char *)strstr(buffer, headers[cur_header]);

    if(toline)
      toline+=strlen(headers[cur_header]);

    for(*toline; toline && *toline!='\n' && cur_rcpt<MAX_RCPTS && cur_pos<MAX_RCPT_ADDR_LEN; toline++){
      switch(*toline)
	{
	case ' ':
	  break;
	case ',':
	case ';':
	  rcpts[cur_rcpt++][cur_pos] = '\0';
	  cur_pos = 0;
	  if(debug_level)
	    fprintf(stdlog, "rcpt: %s\n", rcpts[cur_rcpt-1]);
	  break;
	default:
	  rcpts[cur_rcpt][cur_pos++] = *toline;
	  break;
	}
    }
    
    if(cur_pos>0){
      rcpts[cur_rcpt++][cur_pos] = '\0';
      cur_pos = 0;
      if(debug_level)
	fprintf(stdlog, " %s rcpt: %s\n", headers[cur_header], rcpts[cur_rcpt-1]);
    }
  }
  //should really check for zero rcpts....
  num_rcpts = cur_rcpt;
  if(debug_level)
    fprintf(stdlog, "num rcpt: %d\n", num_rcpts);
  
  return 0;
}

void make_crlf(void)
{
  int i,j;

  memset(crlf_buffer, 0, MAX_MSG_LEN*2+1);

  //crlf_buffer is twice the size of buffer,
  //so even if the entire message was \n's, I won't
  //overflow the new buffer.  TODO: Make this a 
  //little smarter

  for(i=j=0; i<MAX_MSG_LEN; i++,j++){
    if(buffer[i] == '\n' && buffer[i-1] != '\r')
      crlf_buffer[j++] = '\r';
    crlf_buffer[j] = buffer[i];
  }
}

/*
  open socket to host/port
  send data as described above
*/

int send_mail(void)
{
  int i;
  int sock;
  char local_in_buf[BUF_SIZE];
  //*2 to hold the crlf_buffer
  char local_out_buf[MAX_MSG_LEN*2+1];
  int num_bytes;

  if((sock = get_socket()) < 0)
    return 1;

  memset(local_in_buf, 0, BUF_SIZE);

  while(num_bytes = read(sock, local_in_buf, BUF_SIZE-1)){
    if(debug_level)
      fprintf(stdlog, "%s", local_in_buf);
    if((num_bytes < BUF_SIZE-1) || (local_in_buf[BUF_SIZE-2]=='\n'))
      break;
    memset(local_in_buf, 0, BUF_SIZE);
  }  

  snprintf(local_out_buf, strlen(domain)+8, "HELO %s\r\n", domain);
  if(write(sock, local_out_buf, strlen(local_out_buf))<1)
    return 1;

  while(num_bytes = read(sock, local_in_buf, BUF_SIZE-1)){
    if(debug_level)
      fprintf(stdlog, "%s", local_in_buf);
    if((num_bytes < BUF_SIZE-1) || (local_in_buf[BUF_SIZE-2]=='\n'))
      break;
    memset(local_in_buf, 0, BUF_SIZE);
  }  

  snprintf(local_out_buf, strlen(fromaddr)+16, "MAIL FROM: <%s>\r\n", fromaddr);
  if(write(sock, local_out_buf, strlen(local_out_buf))<1)
    return 1;
  
  while(num_bytes = read(sock, local_in_buf, BUF_SIZE-1)){
    if(debug_level)
      fprintf(stdlog, "%s", local_in_buf);
    if((num_bytes < BUF_SIZE-1) || (local_in_buf[BUF_SIZE-2]=='\n'))
      break;
    memset(local_in_buf, 0, BUF_SIZE);
  } 

  for(i=0; i<num_rcpts; i++){
    snprintf(local_out_buf, strlen(rcpts[i])+14, "RCPT TO: <%s>\r\n", rcpts[i]);
    if(write(sock, local_out_buf, strlen(local_out_buf))<1)
      return 1;
  }

  while(num_bytes = read(sock, local_in_buf, BUF_SIZE-1)){
    if(debug_level)
      fprintf(stdlog, "%s", local_in_buf);
    if((num_bytes < BUF_SIZE-1) || (local_in_buf[BUF_SIZE-2]=='\n'))
      break;
    memset(local_in_buf, 0, BUF_SIZE);
  }  

  snprintf(local_out_buf, 7, "DATA\r\n");
  if(write(sock, local_out_buf, strlen(local_out_buf))<1)
    return 1;
  
  while(num_bytes = read(sock, local_in_buf, BUF_SIZE-1)){
    if(debug_level)
      fprintf(stdlog, "%s", local_in_buf);
    if((num_bytes < BUF_SIZE-1) || (local_in_buf[BUF_SIZE-2]=='\n'))
      break;
    memset(local_in_buf, 0, BUF_SIZE);
  } 

  //send the first chunk (that had the headers)
  make_crlf();
  snprintf(local_out_buf, strlen(crlf_buffer)+1, "%s", crlf_buffer);
  //snprintf(local_out_buf, strlen(buffer)+1, "%s", buffer);
  if(write(sock, local_out_buf, strlen(local_out_buf))<1)
    return 1;

  //send any additional data
  while(more_data){
    buffer_mail();
    make_crlf();
    snprintf(local_out_buf, strlen(crlf_buffer)+1, "%s", crlf_buffer);
    //snprintf(local_out_buf, strlen(buffer)+1, "%s", buffer);
    if(write(sock, local_out_buf, strlen(local_out_buf))<1)
      return 1;
  }
							   
  snprintf(local_out_buf, 6, "\r\n.\r\n");
  if(write(sock, local_out_buf, strlen(local_out_buf))<1)
    return 1;
  
  while(num_bytes = read(sock, local_in_buf, BUF_SIZE-1)){
    if(debug_level)
      fprintf(stdlog, "%s", local_in_buf);
    if((num_bytes < BUF_SIZE-1) || (local_in_buf[BUF_SIZE-2]=='\n'))
      break;
    memset(local_in_buf, 0, BUF_SIZE);
  } 

  snprintf(local_out_buf, 7, "QUIT\r\n");
  if(write(sock, local_out_buf, strlen(local_out_buf))<1)
    return 1;
  
  return 0;
}

/* 
   lookup hostname
   open defined port on host
   return socket
*/
int get_socket(void)
{
  struct hostent *servp;
  int sockfd;
  struct sockaddr_in addr;

  servp = gethostbyname(host);
  if(servp == NULL)
    return 0;
  
  if(debug_level){
    fprintf(stdlog,"got name: %s\n",servp->h_name);
    fprintf(stdlog,"got IP: %s\n",inet_ntoa(*((struct in_addr *)servp->h_addr)));
  }  

  if(debug_level > 1){
    sockfd = STDOUT_FILENO;
    
    return sockfd; 
  } else {
    if ((sockfd = socket(AF_INET, SOCK_STREAM, 0)) == -1)
      return 0;
    
    addr.sin_family = AF_INET;
    addr.sin_port = htons(port);
    addr.sin_addr = *((struct in_addr *)servp->h_addr);
    bzero(&(addr.sin_zero), 8); 
    
    if (connect(sockfd, (struct sockaddr *)&addr, sizeof(struct sockaddr)) == -1)
      return 0;
    
    return sockfd;
  }
}



