/* ====================================================================
 * Copyright (c) 1996-2002 The Apache Group.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *	notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *	notice, this list of conditions and the following disclaimer in
 *	the documentation and/or other materials provided with the
 *	distribution.
 *
 * 3. All advertising materials mentioning features or use of this
 *	software must display the following acknowledgment:
 *	"This product includes software developed by the Apache Group
 *	for use in the Apache HTTP server project (http://www.apache.org/)."
 *
 * 4. The names "Apache Server" and "Apache Group" must not be used to
 *	endorse or promote products derived from this software without
 *	prior written permission. For written permission, please contact
 *	apache@apache.org.
 *
 * 5. Products derived from this software may not be called "Apache"
 *	nor may "Apache" appear in their names without prior written
 *	permission of the Apache Group.
 *
 * 6. Redistributions of any form whatsoever must retain the following
 *	acknowledgment:
 *	"This product includes software developed by the Apache Group
 *	for use in the Apache HTTP server project (http://www.apache.org/)."
 *
 * THIS SOFTWARE IS PROVIDED BY THE APACHE GROUP ``AS IS'' AND ANY
 * EXPRESSED OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE APACHE GROUP OR
 * ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 * ====================================================================
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Apache Group and was originally based
 * on public domain software written at the National Center for
 * Supercomputing Applications, University of Illinois, Urbana-Champaign.
 * For more information on the Apache Group and the Apache HTTP server
 * project, please see <http://www.apache.org/>.
 *
 */

/*
 * Improved mod_frontpage
 *
 * Rewritten to enhance and replace the mod_frontpage module supplied
 * by Microsoft and Ready-to-Run Software as part of the Microsoft
 * FrontPage Server Extensions.
 *
 * Gregory A Lundberg <lundberg+apache@vr.net>
 * May 5, 1999
 *
 */

/*
 * Module definition information - the part between the -START and -END
 * lines below is used by Configure. This could be stored in a separate
 * instead.
 *
 * MODULE-DEFINITION-START
 * Name: frontpage_module
 * MODULE-DEFINITION-END
 */

#include "httpd.h"
#include "http_config.h"
#include "http_core.h"
#include "http_log.h"
#include "http_main.h"
#include "http_protocol.h"
#include "util_script.h"
#include "http_conf_globals.h"

#include "mod_frontpage.h"
#include "fpexec.h"
#include "path.h"

module MODULE_VAR_EXPORT frontpage_module;

typedef int boolean;
#undef FALSE
#define FALSE 0

#undef TRUE
#define TRUE  1

typedef struct {
	boolean disabled;
	boolean admindisabled;
} conf;

struct cgi_child_stuff {
	request_rec *r;
	char *argv0;
};

static void
init(server_rec* s, pool* p) {
	ap_add_version_component ("FrontPage/5.0.2.2623");
	while (s != NULL) {
		conf* c = ap_get_module_config (s->module_config, &frontpage_module);
		/*
		 * As default, enable mod_frontpage
		 */
		if (c->disabled == -1)
			c->disabled = FALSE;
		if (c->disabled)
			ap_log_error (APLOG_MARK, APLOG_DEBUG|APLOG_NOERRNO, s,
			    "FrontPage disabled for server %s:%d\n",
			    s->server_hostname, s->port);
		/*
		 * As default, enable the mod_frontpage cgi administration
		 */
		if (c->admindisabled == -1)
			c->admindisabled = TRUE;
		if (c->admindisabled)
			ap_log_error (APLOG_MARK, APLOG_DEBUG|APLOG_NOERRNO, s,
			    "FrontPage CGI-Admin disabled for server %s:%d\n",
			    s->server_hostname, s->port);
		s = s->next;
	}
}

static void*
create_config(pool* p, server_rec* s) {
	conf* new = (conf*) ap_pcalloc (p, sizeof (conf));
	new->disabled = -1;
	new->admindisabled = -1;
	return new;
}

static void*
merge_config(pool* p, void* basev, void* addv) {
	conf* base = (conf*) basev;
	conf* add  = (conf*) addv;
	conf* new  = (conf*) ap_pcalloc (p, sizeof (conf));
	new->disabled = (add->disabled == -1) ?
	    base->disabled : add->disabled;
	new->admindisabled = (add->admindisabled == -1) ?
	    base->admindisabled : add->admindisabled;
	return new;
}

static const char*
disable(cmd_parms* cmd, char* struct_ptr) {
	conf* c = ap_get_module_config (cmd->server->module_config,
	    &frontpage_module);
	c->disabled = TRUE;
	return NULL;
}

static const char*
enable(cmd_parms* cmd, char* struct_ptr) {
	conf* c = ap_get_module_config (cmd->server->module_config,
	    &frontpage_module);
	c->disabled = FALSE;
	return NULL;
}

static const char*
admindisable(cmd_parms* cmd, char* struct_ptr) {
	conf* c = ap_get_module_config (cmd->server->module_config,
	    &frontpage_module);
	c->admindisabled = TRUE;
	return NULL;
}

static const char*
adminenable (cmd_parms* cmd, char* struct_ptr) {
	conf* c = ap_get_module_config (cmd->server->module_config,
	    &frontpage_module);
	c->admindisabled = FALSE;
	return NULL;
}

/*
 * Check if we really have a frontpage request pending or if someone
 * tries to fake it. We do not return OK yet, since we will check for
 * authentification later. Frontpage behaves very ugly since the requests
 * do not even have a similar URI as the filenames have. Often they are
 * in a different place. Nice work.
 */
static int
translate(request_rec *r) {
	char* vti;
	char* sub;
	conf* c = ap_get_module_config (r->server->module_config, &frontpage_module);
	if (c->disabled)
		return DECLINED;
	if ((r->uri[0] != '/') && (r->uri[0] != '\0'))
		return DECLINED;
	/*
	 * If uri does no contain _VTI_BIN, error out.
	 */
	vti = strstr (r->uri, _VTI_BIN);
	if (vti == NULL)
		return DECLINED;
	/*
	 * If the frontpage cgi administration has been disabled,
	 * cancel any requests to fpadmcgi.exe.
	 */
	if ((strstr (vti, ADMINCGI)) || (strstr (vti, ADMIN) != NULL)) {
		if (c->admindisabled)
			return DECLINED;
		else
			r->handler = FRONTPAGE_MAGIC_TYPE;
	} else if ((strstr (vti, AUTHOR  ) != NULL)
	    ||  (strstr (vti, ADMIN   ) != NULL)
	    ||  (strstr (vti, FPCOUNT ) != NULL)
	    ||  (strstr (vti, SHTML   ) != NULL)
	    ||  (strstr (vti, SHTMLDLL) != NULL))
		r->handler = FRONTPAGE_MAGIC_TYPE;
	else if (strstr (vti, VTIHELP) != NULL)
		r->handler = FRONTPAGE_MAGIC_TYPE;
	else if (strrchr (vti, '/') == 0) {
		if (strcmp(sub, "webadmin.css\0") == 0)
			r->handler = FRONTPAGE_MAGIC_TYPE;
	} else if ((sub = strrchr(vti,'.')) != NULL) {
		if ((strcmp(sub, ".css\0") == 0)
		    || (strcmp(sub, ".gif\0") == 0)
		    || (strcmp(sub, ".js\0") == 0)
		    || (strcmp(sub, ".htm\0") == 0))
			r->handler = FRONTPAGE_MAGIC_TYPE;
	}
	return DECLINED;
}

static int
cgi_child(void* child_stuff, child_info* pinfo) {
	struct cgi_child_stuff* cld = (struct cgi_child_stuff*) child_stuff;
	request_rec* r = cld->r;
	char* argv0 = cld->argv0;
	int child_pid;
	int suexec_saved_val;
	char** env;
	RAISE_SIGSTOP (CGI_CHILD);
	ap_add_cgi_vars (r);
	env = ap_create_environment (r->pool, r->subprocess_env);
#ifndef WIN32
	ap_chdir_file (r->filename);
#endif
	ap_error_log2stderr (r->server);
	/* Transumute outselves into the script.
	 * NB only ISINDEX scripts get decoded arguments.
	 */
	ap_cleanup_for_exec ();
	suexec_saved_val = ap_suexec_enabled;
	ap_suexec_enabled = FALSE;
	child_pid = ap_call_exec (r, pinfo, argv0, env, 0);
	ap_suexec_enabled = suexec_saved_val;
#ifdef WIN32
	return (child_pid);
#else
	/* Uh oh.  Still here.  Where's the kaboom?  There was supposed to be an
	 * EARTH-shattering kaboom!
	 *
	 * Oh, well.  Muddle through as best we can...
	 *
	 * Note that only stderr is available at this point, so don't pass in
	 * a server to aplog_error.
	 */
	ap_log_error (APLOG_MARK, APLOG_ERR, NULL, "exec of %s failed", r->filename);
	exit (0);
	/* NOT REACHED */
	return (0);
#endif
}

static int
log_script(request_rec* r, conf* c, int ret,
    char* dbuf, const char* sbuf, BUFF* script_in, BUFF* script_err) {
	char argsbuffer [HUGE_STRING_LEN];
	while (ap_bgets (argsbuffer, HUGE_STRING_LEN, script_in) > 0)
		continue;
	while (ap_bgets (argsbuffer, HUGE_STRING_LEN, script_err) > 0)
		continue;
	return ret;
}

static int
log_scripterror(request_rec *r, conf* c, int ret,
	int show_errno, char *error) {
	ap_log_rerror (APLOG_MARK, show_errno|APLOG_ERR, r,
	    "%s: %s", error, r->filename);
	return ret;
}

static int
frontpage_handler(request_rec* r) {
	char* argv0;
	char* sub;
	char* vti;
	char* webroot;
	char* _vti_pvt;
	uid_t webroot_uid;
	gid_t webroot_gid;
	struct passwd* webroot_pw;
	struct group* webroot_gr;
	boolean NeedAuth = FALSE;
	int retval;
	int IsStatic;
	struct cgi_child_stuff cld;
	BUFF* script_out;
	BUFF* script_in;
	BUFF* script_err;
	char argsbuffer [HUGE_STRING_LEN];
	char* dbuf = NULL;
	conf* c = ap_get_module_config (r->server->module_config, &frontpage_module);

	cld.r = r;
	IsStatic = FALSE;

	if (c->disabled) 
		return DECLINED;

	if (r->method_number == M_OPTIONS) {
		/* 99 out of 100 CGI scripts, this is all they support */
		r->allowed |= (1 << M_GET);
		r->allowed |= (1 << M_POST);
		return DECLINED;
	}

	vti = strstr (r->filename, _VTI_BIN);

	if (vti == NULL)
		return log_scripterror (r, c, NOT_FOUND, APLOG_NOERRNO,
		    "unrecognized FrontPage request");

	if ((c->admindisabled) && ((strstr(vti, ADMINCGI)) != NULL))
		return DECLINED;

	if ((strstr(vti, AUTHOR) != NULL) || (strstr(vti, ADMIN) != NULL) ||
	(strstr(vti, ADMINCGI) != NULL))
		NeedAuth = TRUE;
	else if ((strcmp (vti, FPCOUNT) == 0) || (strcmp (vti, SHTML) == 0) ||
		(strcmp (vti, SHTMLDLL) == 0))
		NeedAuth = FALSE;
	else if ((sub = strrchr(r->uri,'.')) != NULL) {
		if ((strcmp(sub, ".css\0") == 0)
		    || (strcmp(sub, ".gif\0") == 0)
		    || (strcmp(sub, ".js\0") == 0)
		    || (strcmp(sub, ".htm\0") == 0)) {
			NeedAuth = TRUE;
			IsStatic = TRUE;
		}
	} else {
		return log_scripterror (r, c, NOT_FOUND, APLOG_NOERRNO,
		    "unrecognized FrontPage request");
	}

	/*
	 * Overwrite .dll with .exe n vti if a SHTMLDLL
	 * request is pending.
	 */
	if (strcmp(vti, SHTMLDLL) == 0)
	bcopy(SHTML, vti, strlen(SHTML));

	ap_table_set (r->subprocess_env, "FPEXE", ap_pstrdup (r->pool, vti));

	webroot = ap_pstrndup (r->pool, r->filename, (vti - r->filename));
	ap_table_set (r->subprocess_env, "FPDIR", ap_pstrdup (r->pool, webroot));

	_vti_pvt = ap_pstrcat (r->pool, webroot, _VTI_PVT, NULL);

	/*
	 * We *MUST* have been authenticated somehow for AUTHOR or ADMIN requests.
	 * This prevents the single largest hole in FrontPage: if the user somehow
	 * deletes their .htaccess files anyone can gain FrontPage AUTHOR or ADMIN
	 * privileges.  With this check we won't allow ADMIN or AUTHOR unless _some_
	 * authentication was performed.
	 */
	if ((NeedAuth) && 
	    ((r->connection->user == NULL)
	    || (r->connection->ap_auth_type == NULL))) {
		return log_scripterror (r, c, FORBIDDEN, APLOG_NOERRNO,
		    "server configuration did not require authentication");
	}

	/*
	 * Make sure the web root directory exists, is owned by a non-privileged user,
	 * and has proper permissions.
	 */
	errno = 0;
	if (0 != stat (webroot, &r->finfo))
		return log_scripterror (r, c, FORBIDDEN, APLOG_ERR,
		    "cannot stat web root");
	if (!(S_ISDIR (r->finfo.st_mode)))
		return log_scripterror (r, c, FORBIDDEN, APLOG_ERR|APLOG_NOERRNO,
		    "web root not a directory");
	if ((r->finfo.st_uid < FP_UID_MIN)
	||  (r->finfo.st_gid < FP_GID_MIN))
		return log_scripterror (r, c, FORBIDDEN, APLOG_ERR|APLOG_NOERRNO,
		    "web root owned by privileged user");
	if (r->finfo.st_mode & (S_IWGRP | S_IWOTH))
		return log_scripterror (r, c, FORBIDDEN, APLOG_ERR|APLOG_NOERRNO,
		    "web root writable by group or others");
	if (!(r->finfo.st_mode & S_IXOTH))
		return log_scripterror (r, c, FORBIDDEN, APLOG_ERR|APLOG_NOERRNO,
		    "web root not executable by others");
	webroot_uid = r->finfo.st_uid;
	webroot_gid = r->finfo.st_gid;

	errno = 0;
	webroot_pw = getpwuid (webroot_uid);

	if (webroot_pw == NULL) {
		return log_scripterror (r, c, FORBIDDEN, APLOG_ERR,
		    "web owner does not exist");
	}

	ap_table_set (r->subprocess_env, "FPUID", ap_pstrdup (r->pool,
	    webroot_pw->pw_name));

	webroot_gr = getgrgid (webroot_gid);
	if (webroot_gr == NULL) {
		return log_scripterror (r, c, FORBIDDEN, APLOG_ERR,
		    "web group does not exist");
	}

	ap_table_set (r->subprocess_env, "FPGID", ap_pstrdup (r->pool,
	    webroot_gr->gr_name));

	/*
	 * Make sure the _vti_pvt directory exists, is owned by a non-privileged user,
	 * and has proper permissions.
	 */
	errno = 0;
	if (0 != stat (_vti_pvt, &r->finfo)) {
		return log_scripterror (r, c, FORBIDDEN, APLOG_ERR,
		    "cannot stat _vti_pvt");
	}
	if (!(S_ISDIR (r->finfo.st_mode))) {
		return log_scripterror (r, c, FORBIDDEN, APLOG_ERR|APLOG_NOERRNO,
		    "_vti_pvt not a directory");
	}
	if ((r->finfo.st_uid != webroot_uid) 
	||  (r->finfo.st_gid != webroot_gid)) {
		return log_scripterror (r, c, FORBIDDEN, APLOG_ERR|APLOG_NOERRNO,
		    "_vti_pvt not owned by web root owner");
	}
	if (r->finfo.st_mode & (S_IWGRP | S_IWOTH)) {
		return log_scripterror (r, c, FORBIDDEN, APLOG_ERR|APLOG_NOERRNO,
		    "_vti_pvt writable by group or others");
	}
	if (!(r->finfo.st_mode & S_IXOTH)) {
		return log_scripterror (r, c, FORBIDDEN, APLOG_ERR|APLOG_NOERRNO,
		    "_vti_pvt not executable by others");
	}

	/*
	 * Make sure the fpEXEC stub exists, is owned by the correct
	 * user/group, and has proper permissions.
	 */
	if (IsStatic)
		r->filename = FPSTATIC_BIN;
	else
		r->filename = FPEXEC_BIN;
	errno = 0;
	if (0 != stat (r->filename, &r->finfo)) {
		return log_scripterror (r, c, FORBIDDEN, APLOG_ERR,
		    "cannot stat fpEXEC stub");
	}
	if (S_ISLNK (r->finfo.st_mode)) {
		return log_scripterror (r, c, FORBIDDEN, APLOG_ERR|APLOG_NOERRNO,
		    "fpEXEC stub is a symbolic link");
	}
	if (r->finfo.st_uid != 0) {
		return log_scripterror (r, c, FORBIDDEN, APLOG_ERR|APLOG_NOERRNO,
		    "fpEXEC stub not root-owned");
	}
	if (!IsStatic && !(r->finfo.st_mode & S_ISUID)) {
		return log_scripterror (r, c, FORBIDDEN, APLOG_ERR|APLOG_NOERRNO,
		    "fpEXEC stub is not setuid");
	}
	if (r->finfo.st_mode & (S_IWGRP | S_IWOTH)) {
		return log_scripterror (r, c, FORBIDDEN, APLOG_ERR|APLOG_NOERRNO,
		    "fpEXEC stub writable by group or others");
	}
	if (!(r->finfo.st_mode & S_IXOTH)) {
		return log_scripterror (r, c, FORBIDDEN, APLOG_ERR|APLOG_NOERRNO,
		    "fpEXEC stub not executable by others");
	}
	if (((r->finfo.st_mode & S_ISGID))
	&&  (r->finfo.st_gid != 0)) {
		return log_scripterror (r, c, FORBIDDEN, APLOG_ERR|APLOG_NOERRNO,
		    "fpEXEC stub is setgid and not root-group");
	}

	argv0 = strrchr (r->filename, '/');
	if (argv0 != NULL)
		argv0++;
	else
		argv0 = r->filename;

	cld.argv0 = argv0;


	retval = ap_setup_client_block (r, REQUEST_CHUNKED_ERROR);
	if (retval != 0)
		return retval;

	ap_add_common_vars (r);

#ifdef CHARSET_EBCDIC
	/* XXX:@@@ Is the generated/included output ALWAYS in text/ebcdic format? */
	/* Or must we check the Content-Type first? */
	ap_bsetflag (r->connection->client, B_EBCDIC2ASCII, 1);
#endif /*CHARSET_EBCDIC*/

	/*
	 * we spawn out of r->main if it's there so that we can avoid
	 * waiting for free_proc_chain to cleanup in the middle of an
	 * SSI request -djg
	 */
	if (!ap_bspawn_child (r->main ? r->main->pool : r->pool, cgi_child,
	    (void*) &cld, kill_after_timeout,
	    &script_out, &script_in, &script_err)) {
		ap_log_rerror (APLOG_MARK, APLOG_ERR, r,
		    "couldn't spawn child process: %s", r->filename);
		return HTTP_INTERNAL_SERVER_ERROR;
	}

	/* Transfer any put/post args, CERN style...
	 * Note that we already ignore SIGPIPE in the core server.
	 */
	if (ap_should_client_block (r)) {
		int len_read;
		ap_hard_timeout ("copy script args", r);
		while ((len_read = ap_get_client_block (r,
		    argsbuffer, HUGE_STRING_LEN)) > 0) {
			ap_reset_timeout (r);
			if (ap_bwrite (script_out, argsbuffer,
			    len_read) < len_read) {
				/*
				 * silly script stopped reading,
				 * soak up remaining message
				 */
				while (ap_get_client_block (r,
				   argsbuffer, HUGE_STRING_LEN) > 0) {
					/* dump it */
				}
			break;
			}
		}
		ap_bflush (script_out);
		ap_kill_timeout (r);
	}
	ap_bclose (script_out);

	/* Handle script return... */
	if (script_in) {
		const char *location;
		char sbuf[MAX_STRING_LEN];
		int ret = ap_scan_script_header_err_buff (r, script_in, sbuf);
		if (ret != 0)
			return log_script (r, c, ret, dbuf, sbuf, script_in, script_err);
#ifdef CHARSET_EBCDIC
		/* Now check the Content-Type to decide if conversion is needed */
		ap_checkconv (r);
#endif /*CHARSET_EBCDIC*/
		location = ap_table_get (r->headers_out, "Location");
		if ((location	!= NULL)
		    &&  (location[0] == '/' )
		    &&  (r->status   == 200 )) {
			/* Soak up all the script output */
			ap_hard_timeout ("read from script", r);
			while (ap_bgets (argsbuffer, HUGE_STRING_LEN, script_in) > 0)
				continue;
			while (ap_bgets (argsbuffer, HUGE_STRING_LEN, script_err) > 0)
				continue;
			ap_kill_timeout (r);
			/* This redirect needs to be a GET no matter what the original
			 * method was.
			 */
			r->method = ap_pstrdup (r->pool, "GET");
			r->method_number = M_GET;
			/* We already read the message body (if any), so don't allow
			 * the redirected request to think it has one.  We can ignore
			 * Transfer-Encoding, since we used REQUEST_CHUNKED_ERROR.
			 */
			ap_table_unset (r->headers_in, "Content-Length");
			ap_internal_redirect_handler (location, r);
			return OK;
		} else if ((location  != NULL)
		    &&  (r->status == 200 )) {
			/* XX Note that if a script wants to produce its own Redirect
			 * body, it now has to explicitly *say* "Status: 302"
			 */
			return REDIRECT;
		}
		ap_send_http_header (r);
		if (!r->header_only)
			ap_send_fb (script_in, r);
		ap_bclose (script_in);
		ap_soft_timeout ("soaking script stderr", r);
		while (ap_bgets (argsbuffer, HUGE_STRING_LEN, script_err) > 0)
			continue;
		ap_kill_timeout (r);
		ap_bclose (script_err);
	}

	return OK;		/* NOT r->status, even if it has changed. */
}

static const
command_rec cmds[] = {
	{ "FrontPageDisable", disable,
		NULL, RSRC_CONF, NO_ARGS,
		"Disable FrontPage" },
	{ "FrontPageEnable", enable,
		NULL, RSRC_CONF, NO_ARGS,
		"Enable FrontPage" },
	{ "FrontPageAdminDisable", admindisable,
		NULL, RSRC_CONF, NO_ARGS,
		"Disable FrontPageAdmin" },
	{ "FrontPageAdminEnable", adminenable,
		NULL, RSRC_CONF, NO_ARGS,
		"Enable FrontPageAdmin" },
	{ NULL }
};

static const
handler_rec handlers[] = {
	{ FRONTPAGE_MAGIC_TYPE, frontpage_handler },
	{ NULL }
};

module frontpage_module = {
	STANDARD_MODULE_STUFF,
	init,			/* initializer */
	NULL,			/* dir config creator */
	NULL,			/* dir config merger */
	create_config,		/* server config */
	merge_config,		/* server config merger */
	cmds,			/* command table */
	handlers,		/* handlers */
	translate,		/* filename translation */
	NULL,			/* check_user_id */
	NULL,			/* check auth */
	NULL,			/* check access */
	NULL,			/* type_checker */
	NULL,			/* fixups */
	NULL,			/* logger */
#if MODULE_MAGIC_NUMBER >= 19970103
	NULL,			/* header parser */
#endif
#if MODULE_MAGIC_NUMBER >= 19970719
	NULL,			/* child_init */
#endif
#if MODULE_MAGIC_NUMBER >= 19970728
	NULL,			/* child_exit */
#endif
#if MODULE_MAGIC_NUMBER >= 19970902
	NULL			/* post read-request */
#endif
};
