/*
 * nghttp2 - HTTP/2 C Library
 *
 * Copyright (c) 2014 Tatsuhiro Tsujikawa
 *
 * Permission is hereby granted, free of charge, to any person obtaining
 * a copy of this software and associated documentation files (the
 * "Software"), to deal in the Software without restriction, including
 * without limitation the rights to use, copy, modify, merge, publish,
 * distribute, sublicense, and/or sell copies of the Software, and to
 * permit persons to whom the Software is furnished to do so, subject to
 * the following conditions:
 *
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE
 * LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
 * OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
 * WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */
#include "ssl.h"

#include <cassert>
#include <vector>
#include <mutex>
#include <iostream>

#include <openssl/crypto.h>

namespace nghttp2 {

namespace ssl {

// Recommended general purpose "Non-Backward Compatible" cipher by
// mozilla.
//
// https://wiki.mozilla.org/Security/Server_Side_TLS
const char *const DEFAULT_CIPHER_LIST =
    "ECDHE-RSA-AES128-GCM-SHA256:ECDHE-ECDSA-AES128-GCM-SHA256:ECDHE-RSA-"
    "AES256-GCM-SHA384:ECDHE-ECDSA-AES256-GCM-SHA384:DHE-RSA-AES128-GCM-SHA256:"
    "DHE-DSS-AES128-GCM-SHA256:kEDH+AESGCM:ECDHE-RSA-AES128-SHA256:ECDHE-ECDSA-"
    "AES128-SHA256:ECDHE-RSA-AES128-SHA:ECDHE-ECDSA-AES128-SHA:ECDHE-RSA-"
    "AES256-SHA384:ECDHE-ECDSA-AES256-SHA384:ECDHE-RSA-AES256-SHA:ECDHE-ECDSA-"
    "AES256-SHA:DHE-RSA-AES128-SHA256:DHE-RSA-AES128-SHA:DHE-DSS-AES128-SHA256:"
    "DHE-RSA-AES256-SHA256:DHE-DSS-AES256-SHA:DHE-RSA-AES256-SHA:!aNULL:!eNULL:"
    "!EXPORT:!DES:!RC4:!3DES:!MD5:!PSK";

namespace {
std::vector<std::mutex> ssl_global_locks;
} // namespace

namespace {
void ssl_locking_cb(int mode, int type, const char *file, int line) {
  if (mode & CRYPTO_LOCK) {
    ssl_global_locks[type].lock();
  } else {
    ssl_global_locks[type].unlock();
  }
}
} // namespace

LibsslGlobalLock::LibsslGlobalLock() {
  if (!ssl_global_locks.empty()) {
    std::cerr << "OpenSSL global lock has been already set" << std::endl;
    assert(0);
  }
  ssl_global_locks = std::vector<std::mutex>(CRYPTO_num_locks());
  // CRYPTO_set_id_callback(ssl_thread_id); OpenSSL manual says that
  // if threadid_func is not specified using
  // CRYPTO_THREADID_set_callback(), then default implementation is
  // used. We use this default one.
  CRYPTO_set_locking_callback(ssl_locking_cb);
}

LibsslGlobalLock::~LibsslGlobalLock() { ssl_global_locks.clear(); }

const char *get_tls_protocol(SSL *ssl) {
  auto session = SSL_get_session(ssl);
  if (!session) {
    return "unknown";
  }

  switch (session->ssl_version) {
  case SSL2_VERSION:
    return "SSLv2";
  case SSL3_VERSION:
    return "SSLv3";
  case TLS1_2_VERSION:
    return "TLSv1.2";
  case TLS1_1_VERSION:
    return "TLSv1.1";
  case TLS1_VERSION:
    return "TLSv1";
  default:
    return "unknown";
  }
}

TLSSessionInfo *get_tls_session_info(TLSSessionInfo *tls_info, SSL *ssl) {
  if (!ssl) {
    return nullptr;
  }

  auto session = SSL_get_session(ssl);
  if (!session) {
    return nullptr;
  }

  tls_info->cipher = SSL_get_cipher_name(ssl);
  tls_info->protocol = get_tls_protocol(ssl);
  tls_info->session_id = session->session_id;
  tls_info->session_id_length = session->session_id_length;
  tls_info->session_reused = SSL_session_reused(ssl);

  return tls_info;
}

// The black listed cipher suites for HTTP/2 described in RFC 7540.
enum {
  TLS_NULL_WITH_NULL_NULL = 0x0000u,
  TLS_RSA_WITH_NULL_MD5 = 0x0001u,
  TLS_RSA_WITH_NULL_SHA = 0x0002u,
  TLS_RSA_EXPORT_WITH_RC4_40_MD5 = 0x0003u,
  TLS_RSA_WITH_RC4_128_MD5 = 0x0004u,
  TLS_RSA_WITH_RC4_128_SHA = 0x0005u,
  TLS_RSA_EXPORT_WITH_RC2_CBC_40_MD5 = 0x0006u,
  TLS_RSA_WITH_IDEA_CBC_SHA = 0x0007u,
  TLS_RSA_EXPORT_WITH_DES40_CBC_SHA = 0x0008u,
  TLS_RSA_WITH_DES_CBC_SHA = 0x0009u,
  TLS_RSA_WITH_3DES_EDE_CBC_SHA = 0x000Au,
  TLS_DH_DSS_EXPORT_WITH_DES40_CBC_SHA = 0x000Bu,
  TLS_DH_DSS_WITH_DES_CBC_SHA = 0x000Cu,
  TLS_DH_DSS_WITH_3DES_EDE_CBC_SHA = 0x000Du,
  TLS_DH_RSA_EXPORT_WITH_DES40_CBC_SHA = 0x000Eu,
  TLS_DH_RSA_WITH_DES_CBC_SHA = 0x000Fu,
  TLS_DH_RSA_WITH_3DES_EDE_CBC_SHA = 0x0010u,
  TLS_DHE_DSS_EXPORT_WITH_DES40_CBC_SHA = 0x0011u,
  TLS_DHE_DSS_WITH_DES_CBC_SHA = 0x0012u,
  TLS_DHE_DSS_WITH_3DES_EDE_CBC_SHA = 0x0013u,
  TLS_DHE_RSA_EXPORT_WITH_DES40_CBC_SHA = 0x0014u,
  TLS_DHE_RSA_WITH_DES_CBC_SHA = 0x0015u,
  TLS_DHE_RSA_WITH_3DES_EDE_CBC_SHA = 0x0016u,
  TLS_DH_anon_EXPORT_WITH_RC4_40_MD5 = 0x0017u,
  TLS_DH_anon_WITH_RC4_128_MD5 = 0x0018u,
  TLS_DH_anon_EXPORT_WITH_DES40_CBC_SHA = 0x0019u,
  TLS_DH_anon_WITH_DES_CBC_SHA = 0x001Au,
  TLS_DH_anon_WITH_3DES_EDE_CBC_SHA = 0x001Bu,
  TLS_KRB5_WITH_DES_CBC_SHA = 0x001Eu,
  TLS_KRB5_WITH_3DES_EDE_CBC_SHA = 0x001Fu,
  TLS_KRB5_WITH_RC4_128_SHA = 0x0020u,
  TLS_KRB5_WITH_IDEA_CBC_SHA = 0x0021u,
  TLS_KRB5_WITH_DES_CBC_MD5 = 0x0022u,
  TLS_KRB5_WITH_3DES_EDE_CBC_MD5 = 0x0023u,
  TLS_KRB5_WITH_RC4_128_MD5 = 0x0024u,
  TLS_KRB5_WITH_IDEA_CBC_MD5 = 0x0025u,
  TLS_KRB5_EXPORT_WITH_DES_CBC_40_SHA = 0x0026u,
  TLS_KRB5_EXPORT_WITH_RC2_CBC_40_SHA = 0x0027u,
  TLS_KRB5_EXPORT_WITH_RC4_40_SHA = 0x0028u,
  TLS_KRB5_EXPORT_WITH_DES_CBC_40_MD5 = 0x0029u,
  TLS_KRB5_EXPORT_WITH_RC2_CBC_40_MD5 = 0x002Au,
  TLS_KRB5_EXPORT_WITH_RC4_40_MD5 = 0x002Bu,
  TLS_PSK_WITH_NULL_SHA = 0x002Cu,
  TLS_DHE_PSK_WITH_NULL_SHA = 0x002Du,
  TLS_RSA_PSK_WITH_NULL_SHA = 0x002Eu,
  TLS_RSA_WITH_AES_128_CBC_SHA = 0x002Fu,
  TLS_DH_DSS_WITH_AES_128_CBC_SHA = 0x0030u,
  TLS_DH_RSA_WITH_AES_128_CBC_SHA = 0x0031u,
  TLS_DHE_DSS_WITH_AES_128_CBC_SHA = 0x0032u,
  TLS_DHE_RSA_WITH_AES_128_CBC_SHA = 0x0033u,
  TLS_DH_anon_WITH_AES_128_CBC_SHA = 0x0034u,
  TLS_RSA_WITH_AES_256_CBC_SHA = 0x0035u,
  TLS_DH_DSS_WITH_AES_256_CBC_SHA = 0x0036u,
  TLS_DH_RSA_WITH_AES_256_CBC_SHA = 0x0037u,
  TLS_DHE_DSS_WITH_AES_256_CBC_SHA = 0x0038u,
  TLS_DHE_RSA_WITH_AES_256_CBC_SHA = 0x0039u,
  TLS_DH_anon_WITH_AES_256_CBC_SHA = 0x003Au,
  TLS_RSA_WITH_NULL_SHA256 = 0x003Bu,
  TLS_RSA_WITH_AES_128_CBC_SHA256 = 0x003Cu,
  TLS_RSA_WITH_AES_256_CBC_SHA256 = 0x003Du,
  TLS_DH_DSS_WITH_AES_128_CBC_SHA256 = 0x003Eu,
  TLS_DH_RSA_WITH_AES_128_CBC_SHA256 = 0x003Fu,
  TLS_DHE_DSS_WITH_AES_128_CBC_SHA256 = 0x0040u,
  TLS_RSA_WITH_CAMELLIA_128_CBC_SHA = 0x0041u,
  TLS_DH_DSS_WITH_CAMELLIA_128_CBC_SHA = 0x0042u,
  TLS_DH_RSA_WITH_CAMELLIA_128_CBC_SHA = 0x0043u,
  TLS_DHE_DSS_WITH_CAMELLIA_128_CBC_SHA = 0x0044u,
  TLS_DHE_RSA_WITH_CAMELLIA_128_CBC_SHA = 0x0045u,
  TLS_DH_anon_WITH_CAMELLIA_128_CBC_SHA = 0x0046u,
  TLS_DHE_RSA_WITH_AES_128_CBC_SHA256 = 0x0067u,
  TLS_DH_DSS_WITH_AES_256_CBC_SHA256 = 0x0068u,
  TLS_DH_RSA_WITH_AES_256_CBC_SHA256 = 0x0069u,
  TLS_DHE_DSS_WITH_AES_256_CBC_SHA256 = 0x006Au,
  TLS_DHE_RSA_WITH_AES_256_CBC_SHA256 = 0x006Bu,
  TLS_DH_anon_WITH_AES_128_CBC_SHA256 = 0x006Cu,
  TLS_DH_anon_WITH_AES_256_CBC_SHA256 = 0x006Du,
  TLS_RSA_WITH_CAMELLIA_256_CBC_SHA = 0x0084u,
  TLS_DH_DSS_WITH_CAMELLIA_256_CBC_SHA = 0x0085u,
  TLS_DH_RSA_WITH_CAMELLIA_256_CBC_SHA = 0x0086u,
  TLS_DHE_DSS_WITH_CAMELLIA_256_CBC_SHA = 0x0087u,
  TLS_DHE_RSA_WITH_CAMELLIA_256_CBC_SHA = 0x0088u,
  TLS_DH_anon_WITH_CAMELLIA_256_CBC_SHA = 0x0089u,
  TLS_PSK_WITH_RC4_128_SHA = 0x008Au,
  TLS_PSK_WITH_3DES_EDE_CBC_SHA = 0x008Bu,
  TLS_PSK_WITH_AES_128_CBC_SHA = 0x008Cu,
  TLS_PSK_WITH_AES_256_CBC_SHA = 0x008Du,
  TLS_DHE_PSK_WITH_RC4_128_SHA = 0x008Eu,
  TLS_DHE_PSK_WITH_3DES_EDE_CBC_SHA = 0x008Fu,
  TLS_DHE_PSK_WITH_AES_128_CBC_SHA = 0x0090u,
  TLS_DHE_PSK_WITH_AES_256_CBC_SHA = 0x0091u,
  TLS_RSA_PSK_WITH_RC4_128_SHA = 0x0092u,
  TLS_RSA_PSK_WITH_3DES_EDE_CBC_SHA = 0x0093u,
  TLS_RSA_PSK_WITH_AES_128_CBC_SHA = 0x0094u,
  TLS_RSA_PSK_WITH_AES_256_CBC_SHA = 0x0095u,
  TLS_RSA_WITH_SEED_CBC_SHA = 0x0096u,
  TLS_DH_DSS_WITH_SEED_CBC_SHA = 0x0097u,
  TLS_DH_RSA_WITH_SEED_CBC_SHA = 0x0098u,
  TLS_DHE_DSS_WITH_SEED_CBC_SHA = 0x0099u,
  TLS_DHE_RSA_WITH_SEED_CBC_SHA = 0x009Au,
  TLS_DH_anon_WITH_SEED_CBC_SHA = 0x009Bu,
  TLS_RSA_WITH_AES_128_GCM_SHA256 = 0x009Cu,
  TLS_RSA_WITH_AES_256_GCM_SHA384 = 0x009Du,
  TLS_DH_RSA_WITH_AES_128_GCM_SHA256 = 0x00A0u,
  TLS_DH_RSA_WITH_AES_256_GCM_SHA384 = 0x00A1u,
  TLS_DH_DSS_WITH_AES_128_GCM_SHA256 = 0x00A4u,
  TLS_DH_DSS_WITH_AES_256_GCM_SHA384 = 0x00A5u,
  TLS_DH_anon_WITH_AES_128_GCM_SHA256 = 0x00A6u,
  TLS_DH_anon_WITH_AES_256_GCM_SHA384 = 0x00A7u,
  TLS_PSK_WITH_AES_128_GCM_SHA256 = 0x00A8u,
  TLS_PSK_WITH_AES_256_GCM_SHA384 = 0x00A9u,
  TLS_RSA_PSK_WITH_AES_128_GCM_SHA256 = 0x00ACu,
  TLS_RSA_PSK_WITH_AES_256_GCM_SHA384 = 0x00ADu,
  TLS_PSK_WITH_AES_128_CBC_SHA256 = 0x00AEu,
  TLS_PSK_WITH_AES_256_CBC_SHA384 = 0x00AFu,
  TLS_PSK_WITH_NULL_SHA256 = 0x00B0u,
  TLS_PSK_WITH_NULL_SHA384 = 0x00B1u,
  TLS_DHE_PSK_WITH_AES_128_CBC_SHA256 = 0x00B2u,
  TLS_DHE_PSK_WITH_AES_256_CBC_SHA384 = 0x00B3u,
  TLS_DHE_PSK_WITH_NULL_SHA256 = 0x00B4u,
  TLS_DHE_PSK_WITH_NULL_SHA384 = 0x00B5u,
  TLS_RSA_PSK_WITH_AES_128_CBC_SHA256 = 0x00B6u,
  TLS_RSA_PSK_WITH_AES_256_CBC_SHA384 = 0x00B7u,
  TLS_RSA_PSK_WITH_NULL_SHA256 = 0x00B8u,
  TLS_RSA_PSK_WITH_NULL_SHA384 = 0x00B9u,
  TLS_RSA_WITH_CAMELLIA_128_CBC_SHA256 = 0x00BAu,
  TLS_DH_DSS_WITH_CAMELLIA_128_CBC_SHA256 = 0x00BBu,
  TLS_DH_RSA_WITH_CAMELLIA_128_CBC_SHA256 = 0x00BCu,
  TLS_DHE_DSS_WITH_CAMELLIA_128_CBC_SHA256 = 0x00BDu,
  TLS_DHE_RSA_WITH_CAMELLIA_128_CBC_SHA256 = 0x00BEu,
  TLS_DH_anon_WITH_CAMELLIA_128_CBC_SHA256 = 0x00BFu,
  TLS_RSA_WITH_CAMELLIA_256_CBC_SHA256 = 0x00C0u,
  TLS_DH_DSS_WITH_CAMELLIA_256_CBC_SHA256 = 0x00C1u,
  TLS_DH_RSA_WITH_CAMELLIA_256_CBC_SHA256 = 0x00C2u,
  TLS_DHE_DSS_WITH_CAMELLIA_256_CBC_SHA256 = 0x00C3u,
  TLS_DHE_RSA_WITH_CAMELLIA_256_CBC_SHA256 = 0x00C4u,
  TLS_DH_anon_WITH_CAMELLIA_256_CBC_SHA256 = 0x00C5u,
  TLS_EMPTY_RENEGOTIATION_INFO_SCSV = 0x00FFu,
  TLS_ECDH_ECDSA_WITH_NULL_SHA = 0xC001u,
  TLS_ECDH_ECDSA_WITH_RC4_128_SHA = 0xC002u,
  TLS_ECDH_ECDSA_WITH_3DES_EDE_CBC_SHA = 0xC003u,
  TLS_ECDH_ECDSA_WITH_AES_128_CBC_SHA = 0xC004u,
  TLS_ECDH_ECDSA_WITH_AES_256_CBC_SHA = 0xC005u,
  TLS_ECDHE_ECDSA_WITH_NULL_SHA = 0xC006u,
  TLS_ECDHE_ECDSA_WITH_RC4_128_SHA = 0xC007u,
  TLS_ECDHE_ECDSA_WITH_3DES_EDE_CBC_SHA = 0xC008u,
  TLS_ECDHE_ECDSA_WITH_AES_128_CBC_SHA = 0xC009u,
  TLS_ECDHE_ECDSA_WITH_AES_256_CBC_SHA = 0xC00Au,
  TLS_ECDH_RSA_WITH_NULL_SHA = 0xC00Bu,
  TLS_ECDH_RSA_WITH_RC4_128_SHA = 0xC00Cu,
  TLS_ECDH_RSA_WITH_3DES_EDE_CBC_SHA = 0xC00Du,
  TLS_ECDH_RSA_WITH_AES_128_CBC_SHA = 0xC00Eu,
  TLS_ECDH_RSA_WITH_AES_256_CBC_SHA = 0xC00Fu,
  TLS_ECDHE_RSA_WITH_NULL_SHA = 0xC010u,
  TLS_ECDHE_RSA_WITH_RC4_128_SHA = 0xC011u,
  TLS_ECDHE_RSA_WITH_3DES_EDE_CBC_SHA = 0xC012u,
  TLS_ECDHE_RSA_WITH_AES_128_CBC_SHA = 0xC013u,
  TLS_ECDHE_RSA_WITH_AES_256_CBC_SHA = 0xC014u,
  TLS_ECDH_anon_WITH_NULL_SHA = 0xC015u,
  TLS_ECDH_anon_WITH_RC4_128_SHA = 0xC016u,
  TLS_ECDH_anon_WITH_3DES_EDE_CBC_SHA = 0xC017u,
  TLS_ECDH_anon_WITH_AES_128_CBC_SHA = 0xC018u,
  TLS_ECDH_anon_WITH_AES_256_CBC_SHA = 0xC019u,
  TLS_SRP_SHA_WITH_3DES_EDE_CBC_SHA = 0xC01Au,
  TLS_SRP_SHA_RSA_WITH_3DES_EDE_CBC_SHA = 0xC01Bu,
  TLS_SRP_SHA_DSS_WITH_3DES_EDE_CBC_SHA = 0xC01Cu,
  TLS_SRP_SHA_WITH_AES_128_CBC_SHA = 0xC01Du,
  TLS_SRP_SHA_RSA_WITH_AES_128_CBC_SHA = 0xC01Eu,
  TLS_SRP_SHA_DSS_WITH_AES_128_CBC_SHA = 0xC01Fu,
  TLS_SRP_SHA_WITH_AES_256_CBC_SHA = 0xC020u,
  TLS_SRP_SHA_RSA_WITH_AES_256_CBC_SHA = 0xC021u,
  TLS_SRP_SHA_DSS_WITH_AES_256_CBC_SHA = 0xC022u,
  TLS_ECDHE_ECDSA_WITH_AES_128_CBC_SHA256 = 0xC023u,
  TLS_ECDHE_ECDSA_WITH_AES_256_CBC_SHA384 = 0xC024u,
  TLS_ECDH_ECDSA_WITH_AES_128_CBC_SHA256 = 0xC025u,
  TLS_ECDH_ECDSA_WITH_AES_256_CBC_SHA384 = 0xC026u,
  TLS_ECDHE_RSA_WITH_AES_128_CBC_SHA256 = 0xC027u,
  TLS_ECDHE_RSA_WITH_AES_256_CBC_SHA384 = 0xC028u,
  TLS_ECDH_RSA_WITH_AES_128_CBC_SHA256 = 0xC029u,
  TLS_ECDH_RSA_WITH_AES_256_CBC_SHA384 = 0xC02Au,
  TLS_ECDH_ECDSA_WITH_AES_128_GCM_SHA256 = 0xC02Du,
  TLS_ECDH_ECDSA_WITH_AES_256_GCM_SHA384 = 0xC02Eu,
  TLS_ECDH_RSA_WITH_AES_128_GCM_SHA256 = 0xC031u,
  TLS_ECDH_RSA_WITH_AES_256_GCM_SHA384 = 0xC032u,
  TLS_ECDHE_PSK_WITH_RC4_128_SHA = 0xC033u,
  TLS_ECDHE_PSK_WITH_3DES_EDE_CBC_SHA = 0xC034u,
  TLS_ECDHE_PSK_WITH_AES_128_CBC_SHA = 0xC035u,
  TLS_ECDHE_PSK_WITH_AES_256_CBC_SHA = 0xC036u,
  TLS_ECDHE_PSK_WITH_AES_128_CBC_SHA256 = 0xC037u,
  TLS_ECDHE_PSK_WITH_AES_256_CBC_SHA384 = 0xC038u,
  TLS_ECDHE_PSK_WITH_NULL_SHA = 0xC039u,
  TLS_ECDHE_PSK_WITH_NULL_SHA256 = 0xC03Au,
  TLS_ECDHE_PSK_WITH_NULL_SHA384 = 0xC03Bu,
  TLS_RSA_WITH_ARIA_128_CBC_SHA256 = 0xC03Cu,
  TLS_RSA_WITH_ARIA_256_CBC_SHA384 = 0xC03Du,
  TLS_DH_DSS_WITH_ARIA_128_CBC_SHA256 = 0xC03Eu,
  TLS_DH_DSS_WITH_ARIA_256_CBC_SHA384 = 0xC03Fu,
  TLS_DH_RSA_WITH_ARIA_128_CBC_SHA256 = 0xC040u,
  TLS_DH_RSA_WITH_ARIA_256_CBC_SHA384 = 0xC041u,
  TLS_DHE_DSS_WITH_ARIA_128_CBC_SHA256 = 0xC042u,
  TLS_DHE_DSS_WITH_ARIA_256_CBC_SHA384 = 0xC043u,
  TLS_DHE_RSA_WITH_ARIA_128_CBC_SHA256 = 0xC044u,
  TLS_DHE_RSA_WITH_ARIA_256_CBC_SHA384 = 0xC045u,
  TLS_DH_anon_WITH_ARIA_128_CBC_SHA256 = 0xC046u,
  TLS_DH_anon_WITH_ARIA_256_CBC_SHA384 = 0xC047u,
  TLS_ECDHE_ECDSA_WITH_ARIA_128_CBC_SHA256 = 0xC048u,
  TLS_ECDHE_ECDSA_WITH_ARIA_256_CBC_SHA384 = 0xC049u,
  TLS_ECDH_ECDSA_WITH_ARIA_128_CBC_SHA256 = 0xC04Au,
  TLS_ECDH_ECDSA_WITH_ARIA_256_CBC_SHA384 = 0xC04Bu,
  TLS_ECDHE_RSA_WITH_ARIA_128_CBC_SHA256 = 0xC04Cu,
  TLS_ECDHE_RSA_WITH_ARIA_256_CBC_SHA384 = 0xC04Du,
  TLS_ECDH_RSA_WITH_ARIA_128_CBC_SHA256 = 0xC04Eu,
  TLS_ECDH_RSA_WITH_ARIA_256_CBC_SHA384 = 0xC04Fu,
  TLS_RSA_WITH_ARIA_128_GCM_SHA256 = 0xC050u,
  TLS_RSA_WITH_ARIA_256_GCM_SHA384 = 0xC051u,
  TLS_DH_RSA_WITH_ARIA_128_GCM_SHA256 = 0xC054u,
  TLS_DH_RSA_WITH_ARIA_256_GCM_SHA384 = 0xC055u,
  TLS_DH_DSS_WITH_ARIA_128_GCM_SHA256 = 0xC058u,
  TLS_DH_DSS_WITH_ARIA_256_GCM_SHA384 = 0xC059u,
  TLS_DH_anon_WITH_ARIA_128_GCM_SHA256 = 0xC05Au,
  TLS_DH_anon_WITH_ARIA_256_GCM_SHA384 = 0xC05Bu,
  TLS_ECDH_ECDSA_WITH_ARIA_128_GCM_SHA256 = 0xC05Eu,
  TLS_ECDH_ECDSA_WITH_ARIA_256_GCM_SHA384 = 0xC05Fu,
  TLS_ECDH_RSA_WITH_ARIA_128_GCM_SHA256 = 0xC062u,
  TLS_ECDH_RSA_WITH_ARIA_256_GCM_SHA384 = 0xC063u,
  TLS_PSK_WITH_ARIA_128_CBC_SHA256 = 0xC064u,
  TLS_PSK_WITH_ARIA_256_CBC_SHA384 = 0xC065u,
  TLS_DHE_PSK_WITH_ARIA_128_CBC_SHA256 = 0xC066u,
  TLS_DHE_PSK_WITH_ARIA_256_CBC_SHA384 = 0xC067u,
  TLS_RSA_PSK_WITH_ARIA_128_CBC_SHA256 = 0xC068u,
  TLS_RSA_PSK_WITH_ARIA_256_CBC_SHA384 = 0xC069u,
  TLS_PSK_WITH_ARIA_128_GCM_SHA256 = 0xC06Au,
  TLS_PSK_WITH_ARIA_256_GCM_SHA384 = 0xC06Bu,
  TLS_RSA_PSK_WITH_ARIA_128_GCM_SHA256 = 0xC06Eu,
  TLS_RSA_PSK_WITH_ARIA_256_GCM_SHA384 = 0xC06Fu,
  TLS_ECDHE_PSK_WITH_ARIA_128_CBC_SHA256 = 0xC070u,
  TLS_ECDHE_PSK_WITH_ARIA_256_CBC_SHA384 = 0xC071u,
  TLS_ECDHE_ECDSA_WITH_CAMELLIA_128_CBC_SHA256 = 0xC072u,
  TLS_ECDHE_ECDSA_WITH_CAMELLIA_256_CBC_SHA384 = 0xC073u,
  TLS_ECDH_ECDSA_WITH_CAMELLIA_128_CBC_SHA256 = 0xC074u,
  TLS_ECDH_ECDSA_WITH_CAMELLIA_256_CBC_SHA384 = 0xC075u,
  TLS_ECDHE_RSA_WITH_CAMELLIA_128_CBC_SHA256 = 0xC076u,
  TLS_ECDHE_RSA_WITH_CAMELLIA_256_CBC_SHA384 = 0xC077u,
  TLS_ECDH_RSA_WITH_CAMELLIA_128_CBC_SHA256 = 0xC078u,
  TLS_ECDH_RSA_WITH_CAMELLIA_256_CBC_SHA384 = 0xC079u,
  TLS_RSA_WITH_CAMELLIA_128_GCM_SHA256 = 0xC07Au,
  TLS_RSA_WITH_CAMELLIA_256_GCM_SHA384 = 0xC07Bu,
  TLS_DH_RSA_WITH_CAMELLIA_128_GCM_SHA256 = 0xC07Eu,
  TLS_DH_RSA_WITH_CAMELLIA_256_GCM_SHA384 = 0xC07Fu,
  TLS_DH_DSS_WITH_CAMELLIA_128_GCM_SHA256 = 0xC082u,
  TLS_DH_DSS_WITH_CAMELLIA_256_GCM_SHA384 = 0xC083u,
  TLS_DH_anon_WITH_CAMELLIA_128_GCM_SHA256 = 0xC084u,
  TLS_DH_anon_WITH_CAMELLIA_256_GCM_SHA384 = 0xC085u,
  TLS_ECDH_ECDSA_WITH_CAMELLIA_128_GCM_SHA256 = 0xC088u,
  TLS_ECDH_ECDSA_WITH_CAMELLIA_256_GCM_SHA384 = 0xC089u,
  TLS_ECDH_RSA_WITH_CAMELLIA_128_GCM_SHA256 = 0xC08Cu,
  TLS_ECDH_RSA_WITH_CAMELLIA_256_GCM_SHA384 = 0xC08Du,
  TLS_PSK_WITH_CAMELLIA_128_GCM_SHA256 = 0xC08Eu,
  TLS_PSK_WITH_CAMELLIA_256_GCM_SHA384 = 0xC08Fu,
  TLS_RSA_PSK_WITH_CAMELLIA_128_GCM_SHA256 = 0xC092u,
  TLS_RSA_PSK_WITH_CAMELLIA_256_GCM_SHA384 = 0xC093u,
  TLS_PSK_WITH_CAMELLIA_128_CBC_SHA256 = 0xC094u,
  TLS_PSK_WITH_CAMELLIA_256_CBC_SHA384 = 0xC095u,
  TLS_DHE_PSK_WITH_CAMELLIA_128_CBC_SHA256 = 0xC096u,
  TLS_DHE_PSK_WITH_CAMELLIA_256_CBC_SHA384 = 0xC097u,
  TLS_RSA_PSK_WITH_CAMELLIA_128_CBC_SHA256 = 0xC098u,
  TLS_RSA_PSK_WITH_CAMELLIA_256_CBC_SHA384 = 0xC099u,
  TLS_ECDHE_PSK_WITH_CAMELLIA_128_CBC_SHA256 = 0xC09Au,
  TLS_ECDHE_PSK_WITH_CAMELLIA_256_CBC_SHA384 = 0xC09Bu,
  TLS_RSA_WITH_AES_128_CCM = 0xC09Cu,
  TLS_RSA_WITH_AES_256_CCM = 0xC09Du,
  TLS_RSA_WITH_AES_128_CCM_8 = 0xC0A0u,
  TLS_RSA_WITH_AES_256_CCM_8 = 0xC0A1u,
  TLS_PSK_WITH_AES_128_CCM = 0xC0A4u,
  TLS_PSK_WITH_AES_256_CCM = 0xC0A5u,
  TLS_PSK_WITH_AES_128_CCM_8 = 0xC0A8u,
  TLS_PSK_WITH_AES_256_CCM_8 = 0xC0A9u,
};

bool check_http2_requirement(SSL *ssl) {
  auto tls_ver = SSL_version(ssl);

  switch (tls_ver) {
  case TLS1_2_VERSION:
    break;
  default:
    return false;
  }

  auto cipher = SSL_get_current_cipher(ssl);

  // Cipher suites in RFC 7540 balck list are not allowed in HTTP/2.
  switch (SSL_CIPHER_get_id(cipher) & 0xffffu) {
  case TLS_NULL_WITH_NULL_NULL:
  case TLS_RSA_WITH_NULL_MD5:
  case TLS_RSA_WITH_NULL_SHA:
  case TLS_RSA_EXPORT_WITH_RC4_40_MD5:
  case TLS_RSA_WITH_RC4_128_MD5:
  case TLS_RSA_WITH_RC4_128_SHA:
  case TLS_RSA_EXPORT_WITH_RC2_CBC_40_MD5:
  case TLS_RSA_WITH_IDEA_CBC_SHA:
  case TLS_RSA_EXPORT_WITH_DES40_CBC_SHA:
  case TLS_RSA_WITH_DES_CBC_SHA:
  case TLS_RSA_WITH_3DES_EDE_CBC_SHA:
  case TLS_DH_DSS_EXPORT_WITH_DES40_CBC_SHA:
  case TLS_DH_DSS_WITH_DES_CBC_SHA:
  case TLS_DH_DSS_WITH_3DES_EDE_CBC_SHA:
  case TLS_DH_RSA_EXPORT_WITH_DES40_CBC_SHA:
  case TLS_DH_RSA_WITH_DES_CBC_SHA:
  case TLS_DH_RSA_WITH_3DES_EDE_CBC_SHA:
  case TLS_DHE_DSS_EXPORT_WITH_DES40_CBC_SHA:
  case TLS_DHE_DSS_WITH_DES_CBC_SHA:
  case TLS_DHE_DSS_WITH_3DES_EDE_CBC_SHA:
  case TLS_DHE_RSA_EXPORT_WITH_DES40_CBC_SHA:
  case TLS_DHE_RSA_WITH_DES_CBC_SHA:
  case TLS_DHE_RSA_WITH_3DES_EDE_CBC_SHA:
  case TLS_DH_anon_EXPORT_WITH_RC4_40_MD5:
  case TLS_DH_anon_WITH_RC4_128_MD5:
  case TLS_DH_anon_EXPORT_WITH_DES40_CBC_SHA:
  case TLS_DH_anon_WITH_DES_CBC_SHA:
  case TLS_DH_anon_WITH_3DES_EDE_CBC_SHA:
  case TLS_KRB5_WITH_DES_CBC_SHA:
  case TLS_KRB5_WITH_3DES_EDE_CBC_SHA:
  case TLS_KRB5_WITH_RC4_128_SHA:
  case TLS_KRB5_WITH_IDEA_CBC_SHA:
  case TLS_KRB5_WITH_DES_CBC_MD5:
  case TLS_KRB5_WITH_3DES_EDE_CBC_MD5:
  case TLS_KRB5_WITH_RC4_128_MD5:
  case TLS_KRB5_WITH_IDEA_CBC_MD5:
  case TLS_KRB5_EXPORT_WITH_DES_CBC_40_SHA:
  case TLS_KRB5_EXPORT_WITH_RC2_CBC_40_SHA:
  case TLS_KRB5_EXPORT_WITH_RC4_40_SHA:
  case TLS_KRB5_EXPORT_WITH_DES_CBC_40_MD5:
  case TLS_KRB5_EXPORT_WITH_RC2_CBC_40_MD5:
  case TLS_KRB5_EXPORT_WITH_RC4_40_MD5:
  case TLS_PSK_WITH_NULL_SHA:
  case TLS_DHE_PSK_WITH_NULL_SHA:
  case TLS_RSA_PSK_WITH_NULL_SHA:
  case TLS_RSA_WITH_AES_128_CBC_SHA:
  case TLS_DH_DSS_WITH_AES_128_CBC_SHA:
  case TLS_DH_RSA_WITH_AES_128_CBC_SHA:
  case TLS_DHE_DSS_WITH_AES_128_CBC_SHA:
  case TLS_DHE_RSA_WITH_AES_128_CBC_SHA:
  case TLS_DH_anon_WITH_AES_128_CBC_SHA:
  case TLS_RSA_WITH_AES_256_CBC_SHA:
  case TLS_DH_DSS_WITH_AES_256_CBC_SHA:
  case TLS_DH_RSA_WITH_AES_256_CBC_SHA:
  case TLS_DHE_DSS_WITH_AES_256_CBC_SHA:
  case TLS_DHE_RSA_WITH_AES_256_CBC_SHA:
  case TLS_DH_anon_WITH_AES_256_CBC_SHA:
  case TLS_RSA_WITH_NULL_SHA256:
  case TLS_RSA_WITH_AES_128_CBC_SHA256:
  case TLS_RSA_WITH_AES_256_CBC_SHA256:
  case TLS_DH_DSS_WITH_AES_128_CBC_SHA256:
  case TLS_DH_RSA_WITH_AES_128_CBC_SHA256:
  case TLS_DHE_DSS_WITH_AES_128_CBC_SHA256:
  case TLS_RSA_WITH_CAMELLIA_128_CBC_SHA:
  case TLS_DH_DSS_WITH_CAMELLIA_128_CBC_SHA:
  case TLS_DH_RSA_WITH_CAMELLIA_128_CBC_SHA:
  case TLS_DHE_DSS_WITH_CAMELLIA_128_CBC_SHA:
  case TLS_DHE_RSA_WITH_CAMELLIA_128_CBC_SHA:
  case TLS_DH_anon_WITH_CAMELLIA_128_CBC_SHA:
  case TLS_DHE_RSA_WITH_AES_128_CBC_SHA256:
  case TLS_DH_DSS_WITH_AES_256_CBC_SHA256:
  case TLS_DH_RSA_WITH_AES_256_CBC_SHA256:
  case TLS_DHE_DSS_WITH_AES_256_CBC_SHA256:
  case TLS_DHE_RSA_WITH_AES_256_CBC_SHA256:
  case TLS_DH_anon_WITH_AES_128_CBC_SHA256:
  case TLS_DH_anon_WITH_AES_256_CBC_SHA256:
  case TLS_RSA_WITH_CAMELLIA_256_CBC_SHA:
  case TLS_DH_DSS_WITH_CAMELLIA_256_CBC_SHA:
  case TLS_DH_RSA_WITH_CAMELLIA_256_CBC_SHA:
  case TLS_DHE_DSS_WITH_CAMELLIA_256_CBC_SHA:
  case TLS_DHE_RSA_WITH_CAMELLIA_256_CBC_SHA:
  case TLS_DH_anon_WITH_CAMELLIA_256_CBC_SHA:
  case TLS_PSK_WITH_RC4_128_SHA:
  case TLS_PSK_WITH_3DES_EDE_CBC_SHA:
  case TLS_PSK_WITH_AES_128_CBC_SHA:
  case TLS_PSK_WITH_AES_256_CBC_SHA:
  case TLS_DHE_PSK_WITH_RC4_128_SHA:
  case TLS_DHE_PSK_WITH_3DES_EDE_CBC_SHA:
  case TLS_DHE_PSK_WITH_AES_128_CBC_SHA:
  case TLS_DHE_PSK_WITH_AES_256_CBC_SHA:
  case TLS_RSA_PSK_WITH_RC4_128_SHA:
  case TLS_RSA_PSK_WITH_3DES_EDE_CBC_SHA:
  case TLS_RSA_PSK_WITH_AES_128_CBC_SHA:
  case TLS_RSA_PSK_WITH_AES_256_CBC_SHA:
  case TLS_RSA_WITH_SEED_CBC_SHA:
  case TLS_DH_DSS_WITH_SEED_CBC_SHA:
  case TLS_DH_RSA_WITH_SEED_CBC_SHA:
  case TLS_DHE_DSS_WITH_SEED_CBC_SHA:
  case TLS_DHE_RSA_WITH_SEED_CBC_SHA:
  case TLS_DH_anon_WITH_SEED_CBC_SHA:
  case TLS_RSA_WITH_AES_128_GCM_SHA256:
  case TLS_RSA_WITH_AES_256_GCM_SHA384:
  case TLS_DH_RSA_WITH_AES_128_GCM_SHA256:
  case TLS_DH_RSA_WITH_AES_256_GCM_SHA384:
  case TLS_DH_DSS_WITH_AES_128_GCM_SHA256:
  case TLS_DH_DSS_WITH_AES_256_GCM_SHA384:
  case TLS_DH_anon_WITH_AES_128_GCM_SHA256:
  case TLS_DH_anon_WITH_AES_256_GCM_SHA384:
  case TLS_PSK_WITH_AES_128_GCM_SHA256:
  case TLS_PSK_WITH_AES_256_GCM_SHA384:
  case TLS_RSA_PSK_WITH_AES_128_GCM_SHA256:
  case TLS_RSA_PSK_WITH_AES_256_GCM_SHA384:
  case TLS_PSK_WITH_AES_128_CBC_SHA256:
  case TLS_PSK_WITH_AES_256_CBC_SHA384:
  case TLS_PSK_WITH_NULL_SHA256:
  case TLS_PSK_WITH_NULL_SHA384:
  case TLS_DHE_PSK_WITH_AES_128_CBC_SHA256:
  case TLS_DHE_PSK_WITH_AES_256_CBC_SHA384:
  case TLS_DHE_PSK_WITH_NULL_SHA256:
  case TLS_DHE_PSK_WITH_NULL_SHA384:
  case TLS_RSA_PSK_WITH_AES_128_CBC_SHA256:
  case TLS_RSA_PSK_WITH_AES_256_CBC_SHA384:
  case TLS_RSA_PSK_WITH_NULL_SHA256:
  case TLS_RSA_PSK_WITH_NULL_SHA384:
  case TLS_RSA_WITH_CAMELLIA_128_CBC_SHA256:
  case TLS_DH_DSS_WITH_CAMELLIA_128_CBC_SHA256:
  case TLS_DH_RSA_WITH_CAMELLIA_128_CBC_SHA256:
  case TLS_DHE_DSS_WITH_CAMELLIA_128_CBC_SHA256:
  case TLS_DHE_RSA_WITH_CAMELLIA_128_CBC_SHA256:
  case TLS_DH_anon_WITH_CAMELLIA_128_CBC_SHA256:
  case TLS_RSA_WITH_CAMELLIA_256_CBC_SHA256:
  case TLS_DH_DSS_WITH_CAMELLIA_256_CBC_SHA256:
  case TLS_DH_RSA_WITH_CAMELLIA_256_CBC_SHA256:
  case TLS_DHE_DSS_WITH_CAMELLIA_256_CBC_SHA256:
  case TLS_DHE_RSA_WITH_CAMELLIA_256_CBC_SHA256:
  case TLS_DH_anon_WITH_CAMELLIA_256_CBC_SHA256:
  case TLS_EMPTY_RENEGOTIATION_INFO_SCSV:
  case TLS_ECDH_ECDSA_WITH_NULL_SHA:
  case TLS_ECDH_ECDSA_WITH_RC4_128_SHA:
  case TLS_ECDH_ECDSA_WITH_3DES_EDE_CBC_SHA:
  case TLS_ECDH_ECDSA_WITH_AES_128_CBC_SHA:
  case TLS_ECDH_ECDSA_WITH_AES_256_CBC_SHA:
  case TLS_ECDHE_ECDSA_WITH_NULL_SHA:
  case TLS_ECDHE_ECDSA_WITH_RC4_128_SHA:
  case TLS_ECDHE_ECDSA_WITH_3DES_EDE_CBC_SHA:
  case TLS_ECDHE_ECDSA_WITH_AES_128_CBC_SHA:
  case TLS_ECDHE_ECDSA_WITH_AES_256_CBC_SHA:
  case TLS_ECDH_RSA_WITH_NULL_SHA:
  case TLS_ECDH_RSA_WITH_RC4_128_SHA:
  case TLS_ECDH_RSA_WITH_3DES_EDE_CBC_SHA:
  case TLS_ECDH_RSA_WITH_AES_128_CBC_SHA:
  case TLS_ECDH_RSA_WITH_AES_256_CBC_SHA:
  case TLS_ECDHE_RSA_WITH_NULL_SHA:
  case TLS_ECDHE_RSA_WITH_RC4_128_SHA:
  case TLS_ECDHE_RSA_WITH_3DES_EDE_CBC_SHA:
  case TLS_ECDHE_RSA_WITH_AES_128_CBC_SHA:
  case TLS_ECDHE_RSA_WITH_AES_256_CBC_SHA:
  case TLS_ECDH_anon_WITH_NULL_SHA:
  case TLS_ECDH_anon_WITH_RC4_128_SHA:
  case TLS_ECDH_anon_WITH_3DES_EDE_CBC_SHA:
  case TLS_ECDH_anon_WITH_AES_128_CBC_SHA:
  case TLS_ECDH_anon_WITH_AES_256_CBC_SHA:
  case TLS_SRP_SHA_WITH_3DES_EDE_CBC_SHA:
  case TLS_SRP_SHA_RSA_WITH_3DES_EDE_CBC_SHA:
  case TLS_SRP_SHA_DSS_WITH_3DES_EDE_CBC_SHA:
  case TLS_SRP_SHA_WITH_AES_128_CBC_SHA:
  case TLS_SRP_SHA_RSA_WITH_AES_128_CBC_SHA:
  case TLS_SRP_SHA_DSS_WITH_AES_128_CBC_SHA:
  case TLS_SRP_SHA_WITH_AES_256_CBC_SHA:
  case TLS_SRP_SHA_RSA_WITH_AES_256_CBC_SHA:
  case TLS_SRP_SHA_DSS_WITH_AES_256_CBC_SHA:
  case TLS_ECDHE_ECDSA_WITH_AES_128_CBC_SHA256:
  case TLS_ECDHE_ECDSA_WITH_AES_256_CBC_SHA384:
  case TLS_ECDH_ECDSA_WITH_AES_128_CBC_SHA256:
  case TLS_ECDH_ECDSA_WITH_AES_256_CBC_SHA384:
  case TLS_ECDHE_RSA_WITH_AES_128_CBC_SHA256:
  case TLS_ECDHE_RSA_WITH_AES_256_CBC_SHA384:
  case TLS_ECDH_RSA_WITH_AES_128_CBC_SHA256:
  case TLS_ECDH_RSA_WITH_AES_256_CBC_SHA384:
  case TLS_ECDH_ECDSA_WITH_AES_128_GCM_SHA256:
  case TLS_ECDH_ECDSA_WITH_AES_256_GCM_SHA384:
  case TLS_ECDH_RSA_WITH_AES_128_GCM_SHA256:
  case TLS_ECDH_RSA_WITH_AES_256_GCM_SHA384:
  case TLS_ECDHE_PSK_WITH_RC4_128_SHA:
  case TLS_ECDHE_PSK_WITH_3DES_EDE_CBC_SHA:
  case TLS_ECDHE_PSK_WITH_AES_128_CBC_SHA:
  case TLS_ECDHE_PSK_WITH_AES_256_CBC_SHA:
  case TLS_ECDHE_PSK_WITH_AES_128_CBC_SHA256:
  case TLS_ECDHE_PSK_WITH_AES_256_CBC_SHA384:
  case TLS_ECDHE_PSK_WITH_NULL_SHA:
  case TLS_ECDHE_PSK_WITH_NULL_SHA256:
  case TLS_ECDHE_PSK_WITH_NULL_SHA384:
  case TLS_RSA_WITH_ARIA_128_CBC_SHA256:
  case TLS_RSA_WITH_ARIA_256_CBC_SHA384:
  case TLS_DH_DSS_WITH_ARIA_128_CBC_SHA256:
  case TLS_DH_DSS_WITH_ARIA_256_CBC_SHA384:
  case TLS_DH_RSA_WITH_ARIA_128_CBC_SHA256:
  case TLS_DH_RSA_WITH_ARIA_256_CBC_SHA384:
  case TLS_DHE_DSS_WITH_ARIA_128_CBC_SHA256:
  case TLS_DHE_DSS_WITH_ARIA_256_CBC_SHA384:
  case TLS_DHE_RSA_WITH_ARIA_128_CBC_SHA256:
  case TLS_DHE_RSA_WITH_ARIA_256_CBC_SHA384:
  case TLS_DH_anon_WITH_ARIA_128_CBC_SHA256:
  case TLS_DH_anon_WITH_ARIA_256_CBC_SHA384:
  case TLS_ECDHE_ECDSA_WITH_ARIA_128_CBC_SHA256:
  case TLS_ECDHE_ECDSA_WITH_ARIA_256_CBC_SHA384:
  case TLS_ECDH_ECDSA_WITH_ARIA_128_CBC_SHA256:
  case TLS_ECDH_ECDSA_WITH_ARIA_256_CBC_SHA384:
  case TLS_ECDHE_RSA_WITH_ARIA_128_CBC_SHA256:
  case TLS_ECDHE_RSA_WITH_ARIA_256_CBC_SHA384:
  case TLS_ECDH_RSA_WITH_ARIA_128_CBC_SHA256:
  case TLS_ECDH_RSA_WITH_ARIA_256_CBC_SHA384:
  case TLS_RSA_WITH_ARIA_128_GCM_SHA256:
  case TLS_RSA_WITH_ARIA_256_GCM_SHA384:
  case TLS_DH_RSA_WITH_ARIA_128_GCM_SHA256:
  case TLS_DH_RSA_WITH_ARIA_256_GCM_SHA384:
  case TLS_DH_DSS_WITH_ARIA_128_GCM_SHA256:
  case TLS_DH_DSS_WITH_ARIA_256_GCM_SHA384:
  case TLS_DH_anon_WITH_ARIA_128_GCM_SHA256:
  case TLS_DH_anon_WITH_ARIA_256_GCM_SHA384:
  case TLS_ECDH_ECDSA_WITH_ARIA_128_GCM_SHA256:
  case TLS_ECDH_ECDSA_WITH_ARIA_256_GCM_SHA384:
  case TLS_ECDH_RSA_WITH_ARIA_128_GCM_SHA256:
  case TLS_ECDH_RSA_WITH_ARIA_256_GCM_SHA384:
  case TLS_PSK_WITH_ARIA_128_CBC_SHA256:
  case TLS_PSK_WITH_ARIA_256_CBC_SHA384:
  case TLS_DHE_PSK_WITH_ARIA_128_CBC_SHA256:
  case TLS_DHE_PSK_WITH_ARIA_256_CBC_SHA384:
  case TLS_RSA_PSK_WITH_ARIA_128_CBC_SHA256:
  case TLS_RSA_PSK_WITH_ARIA_256_CBC_SHA384:
  case TLS_PSK_WITH_ARIA_128_GCM_SHA256:
  case TLS_PSK_WITH_ARIA_256_GCM_SHA384:
  case TLS_RSA_PSK_WITH_ARIA_128_GCM_SHA256:
  case TLS_RSA_PSK_WITH_ARIA_256_GCM_SHA384:
  case TLS_ECDHE_PSK_WITH_ARIA_128_CBC_SHA256:
  case TLS_ECDHE_PSK_WITH_ARIA_256_CBC_SHA384:
  case TLS_ECDHE_ECDSA_WITH_CAMELLIA_128_CBC_SHA256:
  case TLS_ECDHE_ECDSA_WITH_CAMELLIA_256_CBC_SHA384:
  case TLS_ECDH_ECDSA_WITH_CAMELLIA_128_CBC_SHA256:
  case TLS_ECDH_ECDSA_WITH_CAMELLIA_256_CBC_SHA384:
  case TLS_ECDHE_RSA_WITH_CAMELLIA_128_CBC_SHA256:
  case TLS_ECDHE_RSA_WITH_CAMELLIA_256_CBC_SHA384:
  case TLS_ECDH_RSA_WITH_CAMELLIA_128_CBC_SHA256:
  case TLS_ECDH_RSA_WITH_CAMELLIA_256_CBC_SHA384:
  case TLS_RSA_WITH_CAMELLIA_128_GCM_SHA256:
  case TLS_RSA_WITH_CAMELLIA_256_GCM_SHA384:
  case TLS_DH_RSA_WITH_CAMELLIA_128_GCM_SHA256:
  case TLS_DH_RSA_WITH_CAMELLIA_256_GCM_SHA384:
  case TLS_DH_DSS_WITH_CAMELLIA_128_GCM_SHA256:
  case TLS_DH_DSS_WITH_CAMELLIA_256_GCM_SHA384:
  case TLS_DH_anon_WITH_CAMELLIA_128_GCM_SHA256:
  case TLS_DH_anon_WITH_CAMELLIA_256_GCM_SHA384:
  case TLS_ECDH_ECDSA_WITH_CAMELLIA_128_GCM_SHA256:
  case TLS_ECDH_ECDSA_WITH_CAMELLIA_256_GCM_SHA384:
  case TLS_ECDH_RSA_WITH_CAMELLIA_128_GCM_SHA256:
  case TLS_ECDH_RSA_WITH_CAMELLIA_256_GCM_SHA384:
  case TLS_PSK_WITH_CAMELLIA_128_GCM_SHA256:
  case TLS_PSK_WITH_CAMELLIA_256_GCM_SHA384:
  case TLS_RSA_PSK_WITH_CAMELLIA_128_GCM_SHA256:
  case TLS_RSA_PSK_WITH_CAMELLIA_256_GCM_SHA384:
  case TLS_PSK_WITH_CAMELLIA_128_CBC_SHA256:
  case TLS_PSK_WITH_CAMELLIA_256_CBC_SHA384:
  case TLS_DHE_PSK_WITH_CAMELLIA_128_CBC_SHA256:
  case TLS_DHE_PSK_WITH_CAMELLIA_256_CBC_SHA384:
  case TLS_RSA_PSK_WITH_CAMELLIA_128_CBC_SHA256:
  case TLS_RSA_PSK_WITH_CAMELLIA_256_CBC_SHA384:
  case TLS_ECDHE_PSK_WITH_CAMELLIA_128_CBC_SHA256:
  case TLS_ECDHE_PSK_WITH_CAMELLIA_256_CBC_SHA384:
  case TLS_RSA_WITH_AES_128_CCM:
  case TLS_RSA_WITH_AES_256_CCM:
  case TLS_RSA_WITH_AES_128_CCM_8:
  case TLS_RSA_WITH_AES_256_CCM_8:
  case TLS_PSK_WITH_AES_128_CCM:
  case TLS_PSK_WITH_AES_256_CCM:
  case TLS_PSK_WITH_AES_128_CCM_8:
  case TLS_PSK_WITH_AES_256_CCM_8:
    return false;
  }

  return true;
}

} // namespace ssl

} // namespace nghttp2
