/* virtualmail-pop3d - a POP3 server with virtual domains support
   This code is licensed under the GPL; it has several authors.
   vm-pop3d is based on:
   GNU POP3 - a small, fast, and efficient POP3 daemon
   Copyright (C) 1999 Jakob 'sparky' Kaivo <jkaivo@nodomainname.net>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.  */

#include "vm-pop3d.h"

#ifdef USE_APOP
/* Check if a username exists in APOP password file
   returns pinter to password if found, otherwise NULL */
char *pop3_apopuser(const char *user)
{
    char *password;
    char buf[POP_MAXCMDLEN];
    struct stat st;
#ifdef USE_APOP_BDB
    int errno;
    DB *dbp;
    DBT key, data;
#else
    char *tmp;
    FILE *apop_file;
#endif

#ifdef USE_VIRTUAL
    if (virtualdomain) {
	snprintf(buf, sizeof(buf), "%s/%s/" APOP_PASSFILE,
		 VIRTUAL_PASSWORDS_PATH, virtualdomain);
    } else
#endif
    snprintf(buf, sizeof(buf), APOP_PASSFILE_PATH APOP_PASSFILE);

    if (stat(buf, &st) != -1)
	if ((st.st_mode & 0777) != 0600) {
	    syslog(LOG_INFO, "Bad permissions on APOP password file");
	    return NULL;
	}
#ifdef USE_APOP_BDB
#if DB_VERSION_MAJOR > 2
    if ((errno = db_create(&dbp, NULL, 0)) != 0 || dbp == NULL) {
	syslog(LOG_ERR, "Unable to create database structure: %s",
	       db_strerror(errno));
	return NULL;
    }

    if ((errno = dbp->open(dbp,
#if DB_VERSION_MAJOR >= 4 && DB_VERSION_MINOR >= 1
			   NULL,	/* DB4.1 Transaction index. */
#endif
			   buf, NULL, DB_HASH, DB_RDONLY, 0600)) != 0) {
#else				/* DB_VERSION_MAJOR <= 2 */
    if ((errno = db_open(buf, DB_HASH, DB_RDONLY, 0600, NULL, NULL, &dbp)) != 0) {
#endif
	syslog(LOG_ERR, "Unable to open APOP database %s: %s", buf,
	       strerror(errno));
	return NULL;
    }
    memset(&key, 0, sizeof(DBT));
    memset(&data, 0, sizeof(DBT));

    strncpy(buf, user, sizeof(buf));
    key.data = buf;
    key.size = strlen(user);
    if ((errno = dbp->get(dbp, NULL, &key, &data, 0)) != 0) {
	syslog(LOG_ERR, "db_get error: %s", strerror(errno));
	dbp->close(dbp, 0);
	return NULL;
    }
    if ((password = malloc(sizeof(char) * data.size)) == NULL) {
	dbp->close(dbp, 0);
	return NULL;
    }
    memset(password, 0, (int) data.size);
    memcpy(password, (char *) data.data, (int) data.size);
    dbp->close(dbp, 0);
    return password;
#else				/* !USE_APOP_BDB */
    if ((apop_file = fopen(buf, "r")) == NULL) {
	syslog(LOG_INFO, "Unable to open APOP passwd file %s: %s", buf,
	       strerror(errno));
	return NULL;
    }
    if ((password = malloc(sizeof(char) * APOP_DIGEST)) == NULL) {
	fclose(apop_file);
	pop3_abquit(ERR_NO_MEM);
    }
    password[0] = '\0';
    
    while (fgets(buf, sizeof(buf) - 1, apop_file) != NULL) {
	tmp = index(buf, ':');
	if (!tmp)	/* skip lines without separator */
	    continue;
	*tmp = '\0';
	tmp++;

	if (strcmp(user, buf))
	    continue;

	memcpy(password, tmp, strlen(tmp) + 1);
	tmp = index(password, '\n');
	if (tmp)
	    *tmp = '\0';
	break;
    }

    fclose(apop_file);
    if (*password == '\0') {
	free(password);
	return NULL;
    }
    return password;
#endif				/* BDB2 */

}

extern struct passwd global_pwdentry;

int pop3_apop(const char *arg)
{
    char *tmp, *user_digest, *password;
    struct passwd *pw;
    char buf[POP_MAXCMDLEN];
    struct md5_ctx md5context;
    unsigned char md5digest[16];
    int rc;

    if (state != AUTHORIZATION)
	return ERR_WRONG_STATE;

    if (strlen(arg) == 0)
	return ERR_BAD_ARGS;

    tmp = pop3_cmd(arg);
    if (strlen(tmp) > (POP_MAXCMDLEN - APOP_DIGEST))
	return ERR_BAD_ARGS;

    if ((username = strdup(tmp)) == NULL)
	pop3_abquit(ERR_NO_MEM);
    free(tmp);

#ifdef USE_VIRTUAL
    find_vdomain(username);
#endif

    tmp = pop3_args(arg);
    if (*tmp == '\0' || strlen(tmp) > APOP_DIGEST)
	return ERR_BAD_ARGS;
    if ((user_digest = strdup(tmp)) == NULL)
	pop3_abquit(ERR_NO_MEM);
    free(tmp);

    if ((password = pop3_apopuser(username)) == NULL)
	return ERR_BAD_LOGIN;

    md5_init_ctx(&md5context);
    md5_process_bytes(md5shared, strlen(md5shared), &md5context);
    md5_process_bytes(password, strlen(password), &md5context);
    /* pop3_wipestr (password); */
    free(password);
    md5_finish_ctx(&md5context, md5digest);

    PRINT_HEX(buf, md5digest, 16);
    
    if (strcmp(user_digest, buf))
	return ERR_BAD_LOGIN;

#ifdef USE_VIRTUAL
    if (virtualdomain) {
	/* Setup pw struct for pop3_open_mbox() */
	pw = &global_pwdentry;
	if (uid)
	    pw->pw_uid = uid;
	else
	    pw->pw_uid = (uid_t) VIRTUAL_UID;

	if ((rc = pop3_open_mbox(username, virtualdomain, pw)) != OK)
	    return rc;
    } else {
#endif    
	if ((pw = getpwnam(username)) == NULL)
	    return ERR_BAD_LOGIN;
#ifdef USE_VIRTUAL
	if ((rc = pop3_open_mbox(username, NULL, pw)) != OK)
#else	
	if ((rc = pop3_open_mbox(username, pw)) != OK)
#endif	
	return rc;
#ifdef USE_VIRTUAL
    }
#endif

    state = TRANSACTION;
    pop3_fmt_write(ofile, "+OK opened mailbox for %s\r\n", username);
    cursor = 0;

#ifdef USE_VIRTUAL
    if (virtualdomain)
	syslog(LOG_INFO, "APOP User '%s@%s' logged in from %s, nmsgs=%d",
	       username, virtualdomain, peername, num_messages);
    else
#endif
    syslog(LOG_INFO, "APOP User '%s' logged in from %s, nmsgs=%d",
	   username, peername, num_messages);

    return OK;
}
#endif
