/* virtualmail-pop3d - a POP3 server with virtual domains support
   This code is licensed under the GPL; it has several authors.
   vm-pop3d is based on:
   GNU POP3 - a small, fast, and efficient POP3 daemon
   Copyright (C) 1999 Jakob 'sparky' Kaivo <jkaivo@nodomainname.net>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.  */

#include "vm-pop3d.h"

/* Takes a string as input and returns either the remainder of the string
   after the first space, or a zero length string if no space */

char *pop3_args(const char *cmd)
{
    int space = -1, i = 0, len;
    char *buf;

    len = strlen(cmd) + 1;
    buf = malloc(len * sizeof(char));
    if (buf == NULL)
	pop3_abquit(ERR_NO_MEM);

    while (space < 0 && i < len) {
	if (cmd[i] == ' ')
	    space = i + 1;
	else if (cmd[i] == '\0' || cmd[i] == '\r' || cmd[i] == '\n')
	    len = i;
	i++;
    }

    if (space < 0)
	buf[0] = '\0';
    else {
	for (i = space; i < len; i++)
	    if (cmd[i] == '\0' || cmd[i] == '\r' || cmd[i] == '\n')
		buf[i - space] = '\0';
	    else
		buf[i - space] = cmd[i];
    }

    return buf;
}

/* This takes a string and returns the string up to the first space or end of
   the string, whichever occurs first */

char *pop3_cmd(const char *cmd)
{
    char *buf;
    int i = 0, len;

    len = strlen(cmd) + 1;
    buf = malloc(len * sizeof(char));
    if (buf == NULL)
	pop3_abquit(ERR_NO_MEM);

    for (i = 0; i < len; i++) {
	if (cmd[i] == ' ' || cmd[i] == '\0' || cmd[i] == '\r' || cmd[i] == '\n')
	    len = i;
	else
	    buf[i] = cmd[i];
    }
    buf[i - 1] = '\0';
    return buf;
}

/* Returns OK if a message actually exists and has not been marked by dele(),
   otherwise ERR_NO_MESG */

int pop3_mesg_exist(int mesg)
{
    if ((mesg < 0) || (mesg >= num_messages) || (messages[mesg].deleted == 1))
	return ERR_NO_MESG;

    return OK;
}

/* This is called if it needs to quit without going to the UPDATE stage.
   This is used for conditions such as out of memory, a broken socket, or
   being killed on a signal */

int pop3_abquit(int reason)
{
    pop3_unlock();
    if (mbox){
	fclose(mbox);
	mbox = NULL;
    }
    if (messages){
	free(messages);
	messages = NULL;
    }
    
/* Jonathan Chin <jonathan.chin@ox.compsoc.net>
   Ensure that ofile is non-null so following routine will
   not attempt to write an error message through the socket.
*/

    switch (reason) {
    case ERR_NO_MEM:
	if (ofile != -1)
	    pop3_fmt_write(ofile, "-ERR Out of memory, quitting\r\n");
	syslog(LOG_ERR, "Out of memory");
	break;
    case ERR_DEAD_SOCK:
	if (ofile != -1)	/* should this even print to socket? */
	    pop3_fmt_write(ofile, "-ERR Socket closed, quitting\r\n");
	syslog(LOG_ERR, "Socket closed from %s",
	       (peername) ? peername : "(unknown)");
	break;
    case ERR_SIGNAL:
	if (ofile != -1)
	    pop3_fmt_write(ofile, "-ERR Quitting on signal\r\n");
	break;
    case ERR_TIMEOUT:
	if (ofile != -1)
	    pop3_fmt_write(ofile, "-ERR Session timed out\r\n");
    case ERR_WR_TIMEOUT:
	if (state == TRANSACTION)
	    syslog(LOG_INFO, "Session timed out for user: %s", username);
	else
	    syslog(LOG_INFO, "Session timed out for no user");
	break;
#ifdef IP_BASED_VIRTUAL
    case ERR_IP_BASED:
	if (ofile != -1)
	    pop3_fmt_write(ofile, "-ERR Quitting - host name not found\r\n");
	syslog(LOG_ERR, "Quitting - host name not found");
	break;
#endif				/* IP_BASED_VIRTUAL */
    case ERR_NO_OFILE:
	syslog(LOG_ERR, "Quitting - couldn't open stream");
	break;
    case ERR_MANY_BADCMD:
	if (ofile != -1)
	    pop3_fmt_write(ofile, "-ERR Quitting " MANY_BAD_CMD "\r\n");
	syslog(LOG_ERR, "Quitting " MANY_BAD_CMD);
	break;
    case ERR_MANY_BADPASS:
	if (ofile != -1)
	    pop3_fmt_write(ofile, "-ERR Quitting " MANY_BAD_PASS "\r\n");
	syslog(LOG_ERR, "Quitting " MANY_BAD_PASS);
	break;
    default:
	if (ofile != -1)
	    pop3_fmt_write(ofile, "-ERR Quitting (reason unknown)\r\n");
	syslog(LOG_ERR, "Unknown quit");
	break;
    }

    if (ofile != -1) {
	if (close(ofile))
	    syslog(LOG_ERR, "Error with closing socket: %s", strerror(errno));
	ofile = -1;
    }
    closelog();
    exit(1);
}

/* This locks the mailbox file, works with procmail locking */

int pop3_lock(void)
{
    struct flock fl;
    int lockfd;

    fl.l_type = F_RDLCK;
    fl.l_whence = SEEK_CUR;
    fl.l_start = 0;
    fl.l_len = 0;

    lockfile = malloc(strlen(mailbox) + strlen(".lock") + 1);
    if (lockfile == NULL)
	pop3_abquit(ERR_NO_MEM);
    strcpy(lockfile, mailbox);	/* space allocated immediately above */
    strcat(lockfile, ".lock");

    lockfd = open(lockfile, O_WRONLY | O_EXCL | O_CREAT, 0660);
    if (lockfd == -1) {
	if (errno == EACCES) {
	    return OK;		/* Sliently return OK. No way to lock /dev/null ;) */
	} else if (errno == EEXIST) {
	    free(lockfile);
	    lockfile = NULL;
	    return ERR_MBOX_LOCK;
	}
    }

    if (fcntl(fileno(mbox), F_SETLK, &fl) == -1) {
	syslog(LOG_ERR, "Failed fcntl() file %s lock: %s",
	       lockfile, strerror(errno));
	close(lockfd);
	unlink(lockfile);
	free(lockfile);
	lockfile = NULL;
	return ERR_LOCK_FAILED;
    }
    close(lockfd);
    locked = 1;

    return OK;
}

/* Unlocks the mailbox */

int pop3_unlock(void)
{
    struct flock fl;

    if (locked == 0)
	return OK;

    fl.l_type = F_UNLCK;
    fl.l_whence = SEEK_CUR;
    fl.l_start = 0;
    fl.l_len = 0;

    if (mbox != NULL && fileno(mbox) != -1)
	fcntl(fileno(mbox), F_SETLK, &fl);
    locked = 0;
    if (lockfile != NULL) {
	if (unlink(lockfile) == -1) {
	    syslog(LOG_ERR, "Removing lockfile %s: %s", lockfile,
		   strerror(errno));
	    free(lockfile);
	    lockfile = NULL;
	    return ERR_FILE;
	}
	free(lockfile);
	lockfile = NULL;
    }
    return OK;
}

/* This parses the mailbox to get the number of messages, the size of each
   message, and the read offset of the first character of each message. This
   is where the real speed magic is done */

int pop3_getsizes(void)
{
    char buf[FREAD_SIZE], *p;
    unsigned int max_count = 0;
    /* start with 1 because first line of mbox is From */
    int empty_line = 1, line_count = 0, len = 0;

    struct md5_ctx md5context;

    num_messages = 0;
    messages = NULL;

    if (messages == NULL) {
	messages = malloc(sizeof(message) * 10);
	if (messages == NULL)
	    pop3_abquit(ERR_NO_MEM);
	max_count = 10;
    }
    
    while (fgets(buf, FREAD_SIZE, mbox)) {
	p = memchr(buf, '\n', FREAD_SIZE);
	if (p == NULL)
	    len = FREAD_SIZE - 1;	/* Don't count terminator */
	else
	    len = (p - buf) + 1;	/* count \n too */

	/* Only count a message if "From " is a real mbox delimiter.
	   It depends on the delivery agent; some escape all "From " lines;
	   some don't. */
	if (empty_line == 1 && !strncmp(buf, "From ", 5)) {
	    while (strchr(buf, '\n') == NULL)
		fgets(buf, FREAD_SIZE, mbox);	/* get all of From line */

	    if (num_messages > 0) {	/* another message */
		md5_finish_ctx(&md5context, messages[num_messages - 1].uidl);
	    }
	    num_messages++;

	    line_count = 0;

	    if ((unsigned int) num_messages > max_count) {
		max_count = num_messages * 2;
		messages = realloc(messages, 2 * max_count * sizeof(message));
		if (messages == NULL)
		    pop3_abquit(ERR_NO_MEM);
	    }
#ifdef SEND_FULL_MESSAGE
	    /* Count first 'From ' line too -TEMHOTA */
	    messages[num_messages - 1].header = ftell(mbox) - len;
	    messages[num_messages - 1].size = len + 1;
#else	    
	    messages[num_messages - 1].header = ftell(mbox);
	    messages[num_messages - 1].size = 0;
#endif	    
	    messages[num_messages - 1].deleted = 0;
	    memset(messages[num_messages - 1].uidl, '\0', 17);

	    /* generate md5 hash for UIDL */
	    md5_init_ctx(&md5context);

	} else if (num_messages > 0) {
	    /* Increase only if \n found. This fix possible problem with 
	     * long-long lines in message. -TEMHOTA
	     */
	    if ((len > 0 && len <= FREAD_SIZE - 1) && buf[len - 1] == '\n') {
		len++;
		messages[num_messages - 1].size += len;
	    } else if (len == FREAD_SIZE - 1) {
		/* Long line */
		messages[num_messages - 1].size += len;
	    }
	    /* creating md5 hash for the UIDL using just first 25 lines */
	    /* FiXMe: Should be only for header! -TEMHOTA */
	    if (line_count++ < 26 && len > 0)
		md5_process_bytes(buf, len, &md5context);
	}

	if (buf[0] == '\n')
	    empty_line = 1;
	else
	    empty_line = 0;
    }

    if (num_messages > 0) {
	md5_finish_ctx(&md5context, messages[num_messages - 1].uidl);
    }

    return OK;
}

/* Prints out usage information and exits the program */

void pop3_usage(char *argv0)
{
    printf("Usage: %s [OPTIONS]\n", argv0);
    printf("Runs the vm-pop3d POP3 daemon.\n\n");
#ifdef DEBUG
    printf("  -D, --debug=DEBUGLEVEL   report debugging information\n");
#endif
    printf("  -d, --daemon=MAXCHILDREN runs in daemon mode with a maximum\n");
    printf("                           of MAXCHILDREN child processes\n");
    printf("  -g, --group=name         sets VIRTUAL_GROUP name/gid\n");
    printf("  -h, --help               display this help and exit\n");
    printf("  -i, --inetd              runs in inetd mode (default)\n");
    printf
	("  -p, --port=PORT          specifies alternate port to listen on, implies -d\n");
    printf("  -t, --timeout=TIMEOUT    sets idle timeout to TIMEOUT seconds\n");
    printf("  -u, --user=name          sets VIRTUAL_USER name/uid\n");
    printf("  -v, --version            display version information and exit\n");
}

/* Default signal handler to call the pop3_abquit() function */

void pop3_signal(int sig)
{
    syslog(LOG_ERR, "Quitting on signal: %d", sig);
    
    if (sig == SIGPIPE) {
        /* Silently unlock mailbox and exit. */
	pop3_unlock();
	_exit(254);
    }
    pop3_abquit(ERR_SIGNAL);
}

/* Gets a line of input from the client */

char *pop3_readline(int fd)
{
    fd_set rfds;
    struct timeval tv;
    char *buf;
    int done = 0, rc;
    int too_long = 0;
    int rlen = 0;
    struct timeval tv_now, tv_max;

    if ((buf = malloc(POP_MAXCMDLEN + 1)) == NULL)
	pop3_abquit(ERR_NO_MEM);

    FD_ZERO(&rfds);
    FD_SET(fd, &rfds);

    /* Setup timeout */
    gettimeofday(&tv_max, NULL);
    tv_max.tv_sec += timeout;

    memset(buf, '\0', POP_MAXCMDLEN + 1);

    /* continue to read until newline is received */
    do {
	gettimeofday(&tv_now, NULL);
	if (tv_now.tv_sec >= tv_max.tv_sec)
	    pop3_abquit(ERR_TIMEOUT);

	/* Calculate remain timeout */
	tv.tv_sec = tv_max.tv_sec - tv_now.tv_sec;
	tv.tv_usec = 0;

	rc = select(fd + 1, &rfds, NULL, NULL, &tv);

	if (rc == 0)
	    pop3_abquit(ERR_TIMEOUT);

	if (rc == -1 && errno == EAGAIN)
	    continue;
	/* XXX: FIXME! */

	if ((rlen += read(fd, buf + rlen, POP_MAXCMDLEN - rlen)) < 1)
	    pop3_abquit(ERR_DEAD_SOCK);

	if (rlen == POP_MAXCMDLEN && buf[rlen - 1] != '\n')
	    too_long = 1;
	else if (buf[rlen - 1] == '\n')
	    done = 1;
    }
    while (!done);

    /* only return text if has linefeed in first read() characters */
    if (too_long)
	buf[0] = '\0';

    return buf;
}

/*
 * Format and write line to client
 */

int pop3_fmt_write(int fd, const char *fmt, ...)
{
    va_list ap;
    char buf[1024];
    int len;

    if (fd < 0)			/* Recheck output descriptor again */
	return 0;

    va_start(ap, fmt);
    len = vsnprintf(buf, sizeof(buf), fmt, ap);
    va_end(ap);

    return pop3_cache_write(fd, buf, len);
}

/*
 * Own write buffer. 
 */
static char out_buff[LARGE_BUF_SIZE];
static int buff_ptr = 0;
int debg = 0;

int pop3_cache_write(int fd, char *buf, int len)
{

    /* Cache output in buffer. len = 0 - flush buffer */
    if (len != 0 && ((buff_ptr + len) < LARGE_BUF_SIZE - 1)) {
	memcpy(out_buff + buff_ptr, buf, len);
	buff_ptr += len;
	if (len > 1 && *buf != '+' && *buf != '-' && *buf != '.')
	    return len;		/* Write cached, return success */
	pop3_write(fd, out_buff, buff_ptr);
	buff_ptr = 0;
	out_buff[0] = '\0';
    } else {
	/* Buffer full */
	pop3_write(fd, out_buff, buff_ptr);
	buff_ptr = 0;
	out_buff[0] = '\0';
	/* Check flush request */
	if (len != 0 && *buf != '+' && *buf != '-' && *buf != '.') {
	    memcpy(out_buff + buff_ptr, buf, len);
	    buff_ptr += len;
	} else {
	    pop3_write(fd, buf, len);
	}
    }
    return len;
}

/*
 * Write data to client 
 */
int pop3_write(int fd, char *buf, int len)
{
    fd_set wfds;
    struct timeval tv;
    char *p;
    int wr, rc, tm;

    /* Try write output to client */
    wr = write(fd, buf, len);

    if (wr == len)		/* All ok... */
	return wr;

    p = buf;
    FD_ZERO(&wfds);

    if (timeout == 0)
	tm = 180;
    else
	tm = timeout;

    while (wr != len) {

	FD_SET(fd, &wfds);

	tv.tv_sec = tm;
	tv.tv_usec = 0;

	rc = select(fd + 1, NULL, &wfds, NULL, &tv);
	switch (rc) {
	case -1:
	    if (errno == EINTR || errno == EAGAIN)
		continue;
	    /* XXX: FIXME! */
	    break;

	case 0:
	    pop3_abquit(ERR_WR_TIMEOUT);
	    break;

	default:
	    break;
	}

	if (wr > 0) {
	    p += wr;
	    len -= wr;
	}
	/* Try again write data. */
	wr = write(fd, p, len);
	/* Double fault. */
	if (wr == -1 && errno != EAGAIN) {
	    pop3_abquit(ERR_WR_TIMEOUT);
	}
    }				/* while() */
    return wr;
}

/*
 * Try open user mbox, lock it and parse
 * Input: username, domainname (if need), pw->pw_uid.
 * SideEffect: change dir to spool
 * Modify: mbox/mailbox/username
 * Return: OK/MAILBOX_LOCKED/BAD_LOGIN
 */
#ifdef USE_VIRTUAL
int pop3_open_mbox(const char *user, const char *domain, struct passwd *pw)
#else
int pop3_open_mbox(const char *user, struct passwd *pw)
#endif
{
    struct stat st;

    if (pw != NULL && pw->pw_uid > 1) {
	if (seteuid(pw->pw_uid) == -1) {
	    syslog(LOG_ERR, "seteuid not set: uid %d", pw->pw_uid);
	    return ERR_BAD_LOGIN;
	}
    }
    debug_msg(5, LOG_INFO, "uid %d, gid %d", getuid(), getgid());

    mbox = NULL;

#ifdef MAILSPOOLHOME
#ifdef USE_VIRTUAL
    /* at this time no virtual users can use MAILSPOOLHOME */
    if (!domain) {
#endif				/* USE_VIRTUAL */

	/* TODO: need to check for pw_dir before even wanting to use it */
	mailbox = malloc((strlen(pw->pw_dir) + sizeof(MAILSPOOLHOME) + 1)
			 * sizeof(char));
	if (mailbox == NULL)
	    pop3_abquit(ERR_NO_MEM);
	strcpy(mailbox, pw->pw_dir);
	strcat(mailbox, MAILSPOOLHOME);

	mbox = fopen(mailbox, "r+");
	if (mbox == NULL) {
	    chdir(_PATH_MAILDIR);
	    free(mailbox);
	}
#ifdef USE_VIRTUAL
    }
#endif				/* USE_VIRTUAL */
#endif				/* MAILSPOOLHOME */

#ifdef USE_VIRTUAL
    /* if it has a domainname, it also can't be using MAILSPOOLHOME */
    if (domain) {
	mailbox = malloc((strlen(user) +
			  sizeof(VIRTUAL_MAILPATH) + strlen(domain) +
			  2) * sizeof(char));
	if (mailbox == NULL)
	    pop3_abquit(ERR_NO_MEM);
	strcpy(mailbox, VIRTUAL_MAILPATH);
	strcat(mailbox, "/");
	strcat(mailbox, domain);
	strcat(mailbox, "/");
	chdir(mailbox);
	free(mailbox);
    }
#endif

    if (mbox == NULL) {
	mailbox = strdup(user);	/* the username */
	if (mailbox == NULL)
	    pop3_abquit(ERR_NO_MEM);

	mbox = fopen(mailbox, "r+");
	/* Detect error.. */
	if (mbox == NULL) {
	    if (errno != ENOENT)
		syslog(LOG_ERR, "Unable to use %s's mailbox; Error code=%d",
		       mailbox, errno);
	}
    }

    /* If stat fail - assume empty mailbox */
    if (stat(mailbox, &st) != 0)
	st.st_size = 0;

    /* Lock only opened mbox and not empty. -TEMHOTA */
    if (mbox != NULL && st.st_size != 0) {
	int rc;
	if ((rc = pop3_lock()) != OK) {
	    pop3_unlock();
	    state = AUTHORIZATION;
#ifndef USE_PAM
	    seteuid(0);		/* Back to root privileges 
				 * (allow re-authorize if 
				 * compiled without PAM support)
				 */
#endif
	    return rc;
	}
    }

    username = strdup(mailbox);
    if (username == NULL)
	pop3_abquit(ERR_NO_MEM);

    if (mbox != NULL && st.st_size != 0)
	pop3_getsizes();

    return OK;
}

/*
 * Find virtual domain for this user.
 * May rewrite arg (for apop)
 */
#ifdef USE_VIRTUAL
void find_vdomain(char *arg)
{
    const char domain_delimiters[] = ";!:@%";
    char *temp_domain;

#ifdef IP_BASED_VIRTUAL
    struct hostent *hostent_info;
    struct sockaddr_in name;
    int namelen = sizeof(name);
#endif
    /* reset virtualdomain for every USER attempt */
    if (virtualdomain) {
	free(virtualdomain);
	virtualdomain = NULL;
    }

    /* Some mail clients pre-parse out the username before sending */
    /* so allow colon or at-sign to separate username and domain name */

    temp_domain = strpbrk(arg + 1, domain_delimiters);

    if (temp_domain && (strlen(temp_domain) > 1)) {
	temp_domain[0] = '\0';	/* arg becomes username without hostname */
	temp_domain++;
	if ((virtualdomain = strdup(temp_domain)) == NULL)
	    pop3_abquit(ERR_NO_MEM);
    }

/* placing domain name in USER command overrides IP_BASED_VIRTUAL */
#ifdef IP_BASED_VIRTUAL
    if (virtualdomain == NULL) {
	if (getsockname(ifile,
			(struct sockaddr *) &name,
			(socklen_t *) & namelen) < 0) {
	    /* Allow user from command line -TEMHOTA */
	    if (errno == ENOTSOCK)
		return;
	    debug_msg(3, LOG_ERR, "getsockname: %s", strerror(errno));
	    pop3_abquit(ERR_IP_BASED);
	}
	if (!(temp_domain = (char *) inet_ntoa(name.sin_addr))) {
	    debug_msg(3, LOG_ERR, "inet_ntoa failed: %s", strerror(errno));
	    pop3_abquit(ERR_IP_BASED);
	}
	debug_msg(6, LOG_INFO, "Connection to %s", temp_domain);

	if ((hostent_info = gethostbyaddr((char *) &name.sin_addr, 4, AF_INET))
	    != NULL) {
	    /* virtualdomain must already be NULL (or free) to get here */
	    virtualdomain = malloc(strlen(hostent_info->h_name) + 1);
	    if (virtualdomain == NULL)
		pop3_abquit(ERR_NO_MEM);
	    strncpy(virtualdomain, hostent_info->h_name,
		    strlen(hostent_info->h_name));
	    virtualdomain[strlen(hostent_info->h_name)] = 0;
	    debug_msg(6, LOG_INFO, "hostname: %s", virtualdomain);
	} else {
	    debug_msg(3, LOG_ERR, "gethostbyaddr failure %d for %s",
		      h_errno, temp_domain);
	    pop3_abquit(ERR_IP_BASED);
	}
    }
#endif				/* IP_BASED_VIRTUAL */
}

#endif
