#ifndef GD_H
#define GD_H 1

/* Do the DLL dance: dllexport when building the DLL,
	dllimport when importing from it, nothing when
	not on Silly Silly Windows (tm Aardman Productions). */

/* 2.0.20: for headers */
#ifdef BGDWIN32
#define BGD_EXPORT __declspec(dllexport)
#else
#ifdef WIN32
#define BGD_EXPORT __declspec(dllimport)
#else
/* 2.0.19: should be 'extern', especially for font data pointers */
#define BGD_EXPORT extern
#endif /* WIN32 */
#endif /* BGDWIN32 */

/* 2.0.20: for actual storage of exported data, functions don't need this,
  currently needed only for font pointers */
#ifdef BGDWIN32
#define BGD_EXPORT_DATA_IMPL __declspec(dllexport)
#else
#ifdef WIN32
#define BGD_EXPORT_DATA_IMPL __declspec(dllimport)
#else
/* 2.0.20: should be nothing at all */
#define BGD_EXPORT_DATA_IMPL
#endif /* WIN32 */
#endif /* BGDWIN32 */

#ifdef __cplusplus
extern "C"
{
#endif

/* some might want to set DEFAULT_FONTPATH from configure in config.h */
#ifndef DEFAULT_FONTPATH
/* default fontpath for unix systems  - whatever happened to standards ! */
#define DEFAULT_FONTPATH "/usr/X11R6/lib/X11/fonts/TrueType:/usr/X11R6/lib/X11/fonts/truetype:/usr/X11R6/lib/X11/fonts/TTF:/usr/share/fonts/TrueType:/usr/share/fonts/truetype:/usr/openwin/lib/X11/fonts/TrueType:/usr/X11R6/lib/X11/fonts/Type1"
#endif
#ifndef PATHSEPARATOR
#define PATHSEPARATOR ":"
#endif

/* gd.h: declarations file for the graphic-draw module.
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose and without fee is hereby granted, provided
 * that the above copyright notice appear in all copies and that both that
 * copyright notice and this permission notice appear in supporting
 * documentation.  This software is provided "AS IS." Thomas Boutell and
 * Boutell.Com, Inc. disclaim all warranties, either express or implied, 
 * including but not limited to implied warranties of merchantability and 
 * fitness for a particular purpose, with respect to this code and accompanying
 * documentation. */

/* stdio is needed for file I/O. */
#include <stdio.h>
#include "gd_io.h"

/* The maximum number of palette entries in palette-based images.
	In the wonderful new world of gd 2.0, you can of course have
	many more colors when using truecolor mode. */

#define gdMaxColors 256

/* Image type. See functions below; you will not need to change
	the elements directly. Use the provided macros to
	access sx, sy, the color table, and colorsTotal for 
	read-only purposes. */

/* If 'truecolor' is set true, the image is truecolor; 
	pixels are represented by integers, which
	must be 32 bits wide or more. 

	True colors are repsented as follows:

	ARGB

	Where 'A' (alpha channel) occupies only the
	LOWER 7 BITS of the MSB. This very small 
	loss of alpha channel resolution allows gd 2.x
	to keep backwards compatibility by allowing
	signed integers to be used to represent colors,
	and negative numbers to represent special cases,
	just as in gd 1.x. */

#define gdAlphaMax 127
#define gdAlphaOpaque 0
#define gdAlphaTransparent 127
#define gdRedMax 255
#define gdGreenMax 255
#define gdBlueMax 255
#define gdTrueColorGetAlpha(c) (((c) & 0x7F000000) >> 24)
#define gdTrueColorGetRed(c) (((c) & 0xFF0000) >> 16)
#define gdTrueColorGetGreen(c) (((c) & 0x00FF00) >> 8)
#define gdTrueColorGetBlue(c) ((c) & 0x0000FF)

/* This function accepts truecolor pixel values only. The 
	source color is composited with the destination color
	based on the alpha channel value of the source color.
	The resulting color is opaque. */

   BGD_EXPORT int gdAlphaBlend (int dest, int src);

  typedef struct gdImageStruct
  {
    /* Palette-based image pixels */
    unsigned char **pixels;
    int sx;
    int sy;
    /* These are valid in palette images only. See also
       'alpha', which appears later in the structure to
       preserve binary backwards compatibility */
    int colorsTotal;
    int red[gdMaxColors];
    int green[gdMaxColors];
    int blue[gdMaxColors];
    int open[gdMaxColors];
    /* For backwards compatibility, this is set to the
       first palette entry with 100% transparency,
       and is also set and reset by the 
       gdImageColorTransparent function. Newer
       applications can allocate palette entries
       with any desired level of transparency; however,
       bear in mind that many viewers, notably
       many web browsers, fail to implement
       full alpha channel for PNG and provide
       support for full opacity or transparency only. */
    int transparent;
    int *polyInts;
    int polyAllocated;
    struct gdImageStruct *brush;
    struct gdImageStruct *tile;
    int brushColorMap[gdMaxColors];
    int tileColorMap[gdMaxColors];
    int styleLength;
    int stylePos;
    int *style;
    int interlace;
    /* New in 2.0: thickness of line. Initialized to 1. */
    int thick;
    /* New in 2.0: alpha channel for palettes. Note that only
       Macintosh Internet Explorer and (possibly) Netscape 6
       really support multiple levels of transparency in
       palettes, to my knowledge, as of 2/15/01. Most
       common browsers will display 100% opaque and
       100% transparent correctly, and do something 
       unpredictable and/or undesirable for levels
       in between. TBB */
    int alpha[gdMaxColors];
    /* Truecolor flag and pixels. New 2.0 fields appear here at the
       end to minimize breakage of existing object code. */
    int trueColor;
    int **tpixels;
    /* Should alpha channel be copied, or applied, each time a
       pixel is drawn? This applies to truecolor images only.
       No attempt is made to alpha-blend in palette images,
       even if semitransparent palette entries exist. 
       To do that, build your image as a truecolor image,
       then quantize down to 8 bits. */
    int alphaBlendingFlag;
    /* Should the alpha channel of the image be saved? This affects
       PNG at the moment; other future formats may also
       have that capability. JPEG doesn't. */
    int saveAlphaFlag;

    /* 2.0.12: anti-aliased globals */
    int AA;
    int AA_color;
    int AA_dont_blend;
    unsigned char **AA_opacity;
    int AA_polygon;
    /* Stored and pre-computed variables for determining the perpendicular
       distance from a point to the anti-aliased line being drawn: */
    int AAL_x1;
    int AAL_y1;
    int AAL_x2;
    int AAL_y2;
    int AAL_Bx_Ax;
    int AAL_By_Ay;
    int AAL_LAB_2;
    float AAL_LAB;

    /* 2.0.12: simple clipping rectangle. These values
      must be checked for safety when set; please use
      gdImageSetClip */
    int cx1;
    int cy1;
    int cx2;
    int cy2;
  }
  gdImage;

  typedef gdImage *gdImagePtr;

  typedef struct
  {
    /* # of characters in font */
    int nchars;
    /* First character is numbered... (usually 32 = space) */
    int offset;
    /* Character width and height */
    int w;
    int h;
    /* Font data; array of characters, one row after another.
       Easily included in code, also easily loaded from
       data files. */
    char *data;
  }
  gdFont;

/* Text functions take these. */
  typedef gdFont *gdFontPtr;

/* For backwards compatibility only. Use gdImageSetStyle()
	for MUCH more flexible line drawing. Also see
	gdImageSetBrush(). */
#define gdDashSize 4

/* Special colors. */

#define gdStyled (-2)
#define gdBrushed (-3)
#define gdStyledBrushed (-4)
#define gdTiled (-5)

/* NOT the same as the transparent color index.
	This is used in line styles only. */
#define gdTransparent (-6)

#define gdAntiAliased (-7)

/* Functions to manipulate images. */

/* Creates a palette-based image (up to 256 colors). */
   BGD_EXPORT gdImagePtr gdImageCreate (int sx, int sy);

/* An alternate name for the above (2.0). */
#define gdImageCreatePalette gdImageCreate

/* Creates a truecolor image (millions of colors). */
   BGD_EXPORT gdImagePtr gdImageCreateTrueColor (int sx, int sy);

/* Creates an image from various file types. These functions
	return a palette or truecolor image based on the
	nature of the file being loaded. Truecolor PNG
	stays truecolor; palette PNG stays palette-based;
	JPEG is always truecolor. */
   BGD_EXPORT gdImagePtr gdImageCreateFromPng (FILE * fd);
   BGD_EXPORT gdImagePtr gdImageCreateFromPngCtx (gdIOCtxPtr in);
   BGD_EXPORT gdImagePtr gdImageCreateFromPngPtr (int size, void *data);
   BGD_EXPORT gdImagePtr gdImageCreateFromWBMP (FILE * inFile);
   BGD_EXPORT gdImagePtr gdImageCreateFromWBMPCtx (gdIOCtx * infile);
   BGD_EXPORT gdImagePtr gdImageCreateFromWBMPPtr (int size, void *data);
   BGD_EXPORT gdImagePtr gdImageCreateFromJpeg (FILE * infile);
   BGD_EXPORT gdImagePtr gdImageCreateFromJpegCtx (gdIOCtx * infile);
   BGD_EXPORT gdImagePtr gdImageCreateFromJpegPtr (int size, void *data);

/* A custom data source. */
/* The source function must return -1 on error, otherwise the number
        of bytes fetched. 0 is EOF, not an error! */
/* context will be passed to your source function. */

  typedef struct
  {
    int (*source) (void *context, char *buffer, int len);
    void *context;
  }
  gdSource, *gdSourcePtr;

   /* Deprecated in favor of gdImageCreateFromPngCtx */
   BGD_EXPORT gdImagePtr gdImageCreateFromPngSource (gdSourcePtr in);

#ifdef __FreeBSD__
   BGD_EXPORT gdImagePtr gdImageCreateFromGif(FILE *fd);
   BGD_EXPORT gdImagePtr gdImageCreateFromGifCtx(gdIOCtxPtr in);
   BGD_EXPORT gdImagePtr gdImageCreateFromGifSource(gdSourcePtr in);
#endif

   BGD_EXPORT gdImagePtr gdImageCreateFromGd (FILE * in);
   BGD_EXPORT gdImagePtr gdImageCreateFromGdCtx (gdIOCtxPtr in);
   BGD_EXPORT gdImagePtr gdImageCreateFromGdPtr (int size, void *data);

   BGD_EXPORT gdImagePtr gdImageCreateFromGd2 (FILE * in);
   BGD_EXPORT gdImagePtr gdImageCreateFromGd2Ctx (gdIOCtxPtr in);
   BGD_EXPORT gdImagePtr gdImageCreateFromGd2Ptr (int size, void *data);

   BGD_EXPORT gdImagePtr gdImageCreateFromGd2Part (FILE * in, int srcx, int srcy, int w,
				       int h);
   BGD_EXPORT gdImagePtr gdImageCreateFromGd2PartCtx (gdIOCtxPtr in, int srcx, int srcy,
					  int w, int h);
   BGD_EXPORT gdImagePtr gdImageCreateFromGd2PartPtr (int size, void *data, int srcx, int srcy,
					  int w, int h);
  /* 2.0.10: prototype was missing */
   BGD_EXPORT gdImagePtr gdImageCreateFromXbm (FILE * in);

  /* NOTE: filename, not FILE */
   BGD_EXPORT gdImagePtr gdImageCreateFromXpm (char *filename);

   BGD_EXPORT void gdImageDestroy (gdImagePtr im);

/* Replaces or blends with the background depending on the
	most recent call to gdImageAlphaBlending and the
	alpha channel value of 'color'; default is to overwrite. 
	Tiling and line styling are also implemented
	here. All other gd drawing functions pass through this call, 
	allowing for many useful effects. */

   BGD_EXPORT void gdImageSetPixel (gdImagePtr im, int x, int y, int color);

   BGD_EXPORT int gdImageGetPixel (gdImagePtr im, int x, int y);

   BGD_EXPORT void gdImageAABlend (gdImagePtr im);

   BGD_EXPORT void gdImageLine (gdImagePtr im, int x1, int y1, int x2, int y2, int color);

/* For backwards compatibility only. Use gdImageSetStyle()
	for much more flexible line drawing. */
   BGD_EXPORT void gdImageDashedLine (gdImagePtr im, int x1, int y1, int x2, int y2,
			  int color);
/* Corners specified (not width and height). Upper left first, lower right
 	second. */
   BGD_EXPORT void gdImageRectangle (gdImagePtr im, int x1, int y1, int x2, int y2,
			 int color);
/* Solid bar. Upper left corner first, lower right corner second. */
   BGD_EXPORT void gdImageFilledRectangle (gdImagePtr im, int x1, int y1, int x2, int y2,
			       int color);
   BGD_EXPORT void gdImageSetClip(gdImagePtr im, int x1, int y1, int x2, int y2);
   BGD_EXPORT void gdImageGetClip(gdImagePtr im, int *x1P, int *y1P, int *x2P, int *y2P);
   BGD_EXPORT int gdImageBoundsSafe (gdImagePtr im, int x, int y);
   BGD_EXPORT void gdImageChar (gdImagePtr im, gdFontPtr f, int x, int y, int c,
		    int color);
   BGD_EXPORT void gdImageCharUp (gdImagePtr im, gdFontPtr f, int x, int y, int c,
		      int color);
   BGD_EXPORT void gdImageString (gdImagePtr im, gdFontPtr f, int x, int y,
		      unsigned char *s, int color);
   BGD_EXPORT void gdImageStringUp (gdImagePtr im, gdFontPtr f, int x, int y,
			unsigned char *s, int color);
   BGD_EXPORT void gdImageString16 (gdImagePtr im, gdFontPtr f, int x, int y,
			unsigned short *s, int color);
   BGD_EXPORT void gdImageStringUp16 (gdImagePtr im, gdFontPtr f, int x, int y,
			  unsigned short *s, int color);

/* 2.0.16: for thread-safe use of gdImageStringFT and friends,
  call this before allowing any thread to call gdImageStringFT. 
  Otherwise it is invoked by the first thread to invoke
  gdImageStringFT, with a very small but real risk of a race condition. 
  Return 0 on success, nonzero on failure to initialize freetype. */
 BGD_EXPORT int gdFontCacheSetup (void);

/* Optional: clean up after application is done using fonts in 
 BGD_EXPORT   gdImageStringFT(). */
 BGD_EXPORT void gdFontCacheShutdown (void);
/* 2.0.20: for backwards compatibility. A few applications did start calling
 this function when it first appeared although it was never documented. 
 Simply invokes gdFontCacheShutdown. */
 BGD_EXPORT void gdFreeFontCache (void);

/* Calls gdImageStringFT. Provided for backwards compatibility only. */
  BGD_EXPORT char *gdImageStringTTF (gdImage * im, int *brect, int fg, char *fontlist,
			  double ptsize, double angle, int x, int y,
			  char *string);

/* FreeType 2 text output */
  BGD_EXPORT char *gdImageStringFT (gdImage * im, int *brect, int fg, char *fontlist,
			 double ptsize, double angle, int x, int y,
			 char *string);

/* 2.0.5: provides an extensible way to pass additional parameters.
	Thanks to Wez Furlong, sorry for the delay. */

  typedef struct
  {
    int flags;			/* Logical OR of gdFTEX_ values */
    double linespacing;		/* fine tune line spacing for '\n' */
    int charmap;		/* TBB: 2.0.12: may be gdFTEX_Unicode,
				   gdFTEX_Shift_JIS, or gdFTEX_Big5;
				   when not specified, maps are searched
				   for in the above order. */
  }
  gdFTStringExtra, *gdFTStringExtraPtr;

#define gdFTEX_LINESPACE 1
#define gdFTEX_CHARMAP 2

/* These are NOT flags; set one in 'charmap' if you set the
	gdFTEX_CHARMAP bit in 'flags'. */
#define gdFTEX_Unicode 0
#define gdFTEX_Shift_JIS 1
#define gdFTEX_Big5 2

  BGD_EXPORT char *gdImageStringFTEx (gdImage * im, int *brect, int fg, char *fontlist,
			   double ptsize, double angle, int x, int y,
			   char *string, gdFTStringExtraPtr strex);

/* Point type for use in polygon drawing. */
  typedef struct
  {
    int x, y;
  }
  gdPoint, *gdPointPtr;

   BGD_EXPORT void gdImagePolygon (gdImagePtr im, gdPointPtr p, int n, int c);
#ifdef __FreeBSD__
   BGD_EXPORT void gdImageOpenPolygon(gdImagePtr im, gdPointPtr p, int n, int c);
#endif
   BGD_EXPORT void gdImageFilledPolygon (gdImagePtr im, gdPointPtr p, int n, int c);

/* These functions still work with truecolor images, 
	for which they never return error. */
   BGD_EXPORT int gdImageColorAllocate (gdImagePtr im, int r, int g, int b);
/* gd 2.0: palette entries with non-opaque transparency are permitted. */
   BGD_EXPORT int gdImageColorAllocateAlpha (gdImagePtr im, int r, int g, int b, int a);
/* Assumes opaque is the preferred alpha channel value */
   BGD_EXPORT int gdImageColorClosest (gdImagePtr im, int r, int g, int b);
/* Closest match taking all four parameters into account.
	A slightly different color with the same transparency
	beats the exact same color with radically different
	transparency */
   BGD_EXPORT int gdImageColorClosestAlpha (gdImagePtr im, int r, int g, int b, int a);
/* An alternate method */
   BGD_EXPORT int gdImageColorClosestHWB (gdImagePtr im, int r, int g, int b);
/* Returns exact, 100% opaque matches only */
   BGD_EXPORT int gdImageColorExact (gdImagePtr im, int r, int g, int b);
/* Returns an exact match only, including alpha */
   BGD_EXPORT int gdImageColorExactAlpha (gdImagePtr im, int r, int g, int b, int a);
/* Opaque only */
   BGD_EXPORT int gdImageColorResolve (gdImagePtr im, int r, int g, int b);
/* Based on gdImageColorExactAlpha and gdImageColorClosestAlpha */
   BGD_EXPORT int gdImageColorResolveAlpha (gdImagePtr im, int r, int g, int b, int a);

/* A simpler way to obtain an opaque truecolor value for drawing on a
	truecolor image. Not for use with palette images! */

#define gdTrueColor(r, g, b) (((r) << 16) + \
	((g) << 8) + \
	(b))

/* Returns a truecolor value with an alpha channel component.
	gdAlphaMax (127, **NOT 255**) is transparent, 0 is completely
	opaque. */

#define gdTrueColorAlpha(r, g, b, a) (((a) << 24) + \
	((r) << 16) + \
	((g) << 8) + \
	(b))

   BGD_EXPORT void gdImageColorDeallocate (gdImagePtr im, int color);

/* Converts a truecolor image to a palette-based image,
	using a high-quality two-pass quantization routine
	which attempts to preserve alpha channel information
	as well as R/G/B color information when creating
	a palette. If ditherFlag is set, the image will be
	dithered to approximate colors better, at the expense
	of some obvious "speckling." colorsWanted can be
	anything up to 256. If the original source image
	includes photographic information or anything that
	came out of a JPEG, 256 is strongly recommended.

	Better yet, don't use this function -- write real
	truecolor PNGs and JPEGs. The disk space gain of
        conversion to palette is not great (for small images
        it can be negative) and the quality loss is ugly. */

   BGD_EXPORT void gdImageTrueColorToPalette (gdImagePtr im, int ditherFlag,
				  int colorsWanted);

/* Specifies a color index (if a palette image) or an
	RGB color (if a truecolor image) which should be
	considered 100% transparent. FOR TRUECOLOR IMAGES,
	THIS IS IGNORED IF AN ALPHA CHANNEL IS BEING
	SAVED. Use gdImageSaveAlpha(im, 0); to
	turn off the saving of a full alpha channel in
	a truecolor image. Note that gdImageColorTransparent
	is usually compatible with older browsers that
	do not understand full alpha channels well. TBB */
   BGD_EXPORT void gdImageColorTransparent (gdImagePtr im, int color);

   BGD_EXPORT void gdImagePaletteCopy (gdImagePtr dst, gdImagePtr src);
   BGD_EXPORT void gdImagePng (gdImagePtr im, FILE * out);
   BGD_EXPORT void gdImagePngCtx (gdImagePtr im, gdIOCtx * out);
  /* 2.0.12: Compression level: 0-9 or -1, where 0 is NO COMPRESSION at all,
     1 is FASTEST but produces larger files, 9 provides the best
     compression (smallest files) but takes a long time to compress, and
     -1 selects the default compiled into the zlib library. */
   BGD_EXPORT void gdImagePngEx (gdImagePtr im, FILE * out, int level);
   BGD_EXPORT void gdImagePngCtxEx (gdImagePtr im, gdIOCtx * out, int level);

   BGD_EXPORT void gdImageWBMP (gdImagePtr image, int fg, FILE * out);
   BGD_EXPORT void gdImageWBMPCtx (gdImagePtr image, int fg, gdIOCtx * out);

/* Guaranteed to correctly free memory returned
	by the gdImage*Ptr functions */
   BGD_EXPORT void gdFree (void *m);

/* Best to free this memory with gdFree(), not free() */
  void *gdImageWBMPPtr (gdImagePtr im, int *size, int fg);

/* 100 is highest quality (there is always a little loss with JPEG).
	0 is lowest. 10 is about the lowest useful setting. */
   BGD_EXPORT void gdImageJpeg (gdImagePtr im, FILE * out, int quality);
   BGD_EXPORT void gdImageJpegCtx (gdImagePtr im, gdIOCtx * out, int quality);

/* Best to free this memory with gdFree(), not free() */
  BGD_EXPORT void *gdImageJpegPtr (gdImagePtr im, int *size, int quality);

#ifdef __FreeBSD__
  BGD_EXPORT void gdImageLzw(gdImagePtr im, FILE *out);
  BGD_EXPORT void* gdImageLzwPtr(gdImagePtr im, int *size);
  BGD_EXPORT void gdImageLzwCtx(gdImagePtr im, gdIOCtxPtr out);

  BGD_EXPORT void gdImageBigGif(gdImagePtr im, FILE *out);
  BGD_EXPORT void* gdImageBigGifPtr(gdImagePtr im, int *size);
  BGD_EXPORT void gdImageBigGifCtx(gdImagePtr im, gdIOCtxPtr out);
#endif


/* A custom data sink. For backwards compatibility. Use
	gdIOCtx instead. */
/* The sink function must return -1 on error, otherwise the number
        of bytes written, which must be equal to len. */
/* context will be passed to your sink function. */
  typedef struct
  {
    int (*sink) (void *context, const char *buffer, int len);
    void *context;
  }
  gdSink, *gdSinkPtr;

   BGD_EXPORT void gdImagePngToSink (gdImagePtr im, gdSinkPtr out);

#ifdef __FreeBSD__
   BGD_EXPORT void gdImageGif(gdImagePtr im, FILE *out);
   BGD_EXPORT void* gdImageGifPtr(gdImagePtr im, int *size);
   BGD_EXPORT void gdImageGifCtx(gdImagePtr im, gdIOCtxPtr out);
   BGD_EXPORT void gdImageGifToSink(gdImagePtr im, gdSinkPtr out);
#endif

   BGD_EXPORT void gdImageGd (gdImagePtr im, FILE * out);
   BGD_EXPORT void gdImageGd2 (gdImagePtr im, FILE * out, int cs, int fmt);

/* Best to free this memory with gdFree(), not free() */
  BGD_EXPORT void *gdImagePngPtr (gdImagePtr im, int *size);
  BGD_EXPORT void *gdImagePngPtrEx (gdImagePtr im, int *size, int level);

/* Best to free this memory with gdFree(), not free() */
  BGD_EXPORT void *gdImageGdPtr (gdImagePtr im, int *size);

/* Best to free this memory with gdFree(), not free() */
  BGD_EXPORT void *gdImageGd2Ptr (gdImagePtr im, int cs, int fmt, int *size);

   BGD_EXPORT void gdImageEllipse (gdImagePtr im, int cx, int cy, int w, int h,
		       int color);

/* Style is a bitwise OR ( | operator ) of these.
	gdArc and gdChord are mutually exclusive;
	gdChord just connects the starting and ending
	angles with a straight line, while gdArc produces
	a rounded edge. gdPie is a synonym for gdArc. 
	gdNoFill indicates that the arc or chord should be
	outlined, not filled. gdEdged, used together with
	gdNoFill, indicates that the beginning and ending
	angles should be connected to the center; this is
	a good way to outline (rather than fill) a
	'pie slice'. */
#define gdArc   0
#define gdPie   gdArc
#define gdChord 1
#define gdNoFill 2
#define gdEdged 4

   BGD_EXPORT void gdImageFilledArc (gdImagePtr im, int cx, int cy, int w, int h, int s,
			 int e, int color, int style);
   BGD_EXPORT void gdImageArc (gdImagePtr im, int cx, int cy, int w, int h, int s, int e,
		   int color);
   BGD_EXPORT void gdImageFilledEllipse (gdImagePtr im, int cx, int cy, int w, int h,
			     int color);
   BGD_EXPORT void gdImageFillToBorder (gdImagePtr im, int x, int y, int border,
			    int color);
   BGD_EXPORT void gdImageFill (gdImagePtr im, int x, int y, int color);
   BGD_EXPORT void gdImageCopy (gdImagePtr dst, gdImagePtr src, int dstX, int dstY,
		    int srcX, int srcY, int w, int h);
   BGD_EXPORT void gdImageCopyMerge (gdImagePtr dst, gdImagePtr src, int dstX, int dstY,
			 int srcX, int srcY, int w, int h, int pct);
   BGD_EXPORT void gdImageCopyMergeGray (gdImagePtr dst, gdImagePtr src, int dstX,
			     int dstY, int srcX, int srcY, int w, int h,
			     int pct);

/* Stretches or shrinks to fit, as needed. Does NOT attempt
	to average the entire set of source pixels that scale down onto the
	destination pixel. */
   BGD_EXPORT void gdImageCopyResized (gdImagePtr dst, gdImagePtr src, int dstX, int dstY,
			   int srcX, int srcY, int dstW, int dstH, int srcW,
			   int srcH);

/* gd 2.0: stretches or shrinks to fit, as needed. When called with a
	truecolor destination image, this function averages the
	entire set of source pixels that scale down onto the
	destination pixel, taking into account what portion of the
	destination pixel each source pixel represents. This is a
	floating point operation, but this is not a performance issue
	on modern hardware, except for some embedded devices. If the 
	destination is a palette image, gdImageCopyResized is 
	substituted automatically. */
   BGD_EXPORT void gdImageCopyResampled (gdImagePtr dst, gdImagePtr src, int dstX,
			     int dstY, int srcX, int srcY, int dstW, int dstH,
			     int srcW, int srcH);

/* gd 2.0.8: gdImageCopyRotated is added. Source
        is a rectangle, with its upper left corner at
        srcX and srcY. Destination is the *center* of
        the rotated copy. Angle is in degrees, same as
        gdImageArc. Floating point destination center
        coordinates allow accurate rotation of
        objects of odd-numbered width or height. */
   BGD_EXPORT void gdImageCopyRotated (gdImagePtr dst,
			   gdImagePtr src,
			   double dstX, double dstY,
			   int srcX, int srcY,
			   int srcWidth, int srcHeight, int angle);

   BGD_EXPORT void gdImageSetBrush (gdImagePtr im, gdImagePtr brush);
   BGD_EXPORT void gdImageSetTile (gdImagePtr im, gdImagePtr tile);
   BGD_EXPORT void gdImageSetAntiAliased (gdImagePtr im, int c);
   BGD_EXPORT void gdImageSetAntiAliasedDontBlend (gdImagePtr im, int c, int dont_blend);
   BGD_EXPORT void gdImageSetStyle (gdImagePtr im, int *style, int noOfPixels);
/* Line thickness (defaults to 1). Affects lines, ellipses, 
	rectangles, polygons and so forth. */
   BGD_EXPORT void gdImageSetThickness (gdImagePtr im, int thickness);
/* On or off (1 or 0) for all three of these. */
   BGD_EXPORT void gdImageInterlace (gdImagePtr im, int interlaceArg);
   BGD_EXPORT void gdImageAlphaBlending (gdImagePtr im, int alphaBlendingArg);
   BGD_EXPORT void gdImageSaveAlpha (gdImagePtr im, int saveAlphaArg);

/* Macros to access information about images. */

/* Returns nonzero if the image is a truecolor image,
	zero for a palette image. */

#define gdImageTrueColor(im) ((im)->trueColor)

#define gdImageSX(im) ((im)->sx)
#define gdImageSY(im) ((im)->sy)
#define gdImageColorsTotal(im) ((im)->colorsTotal)
#define gdImageRed(im, c) ((im)->trueColor ? gdTrueColorGetRed(c) : \
	(im)->red[(c)])
#define gdImageGreen(im, c) ((im)->trueColor ? gdTrueColorGetGreen(c) : \
	(im)->green[(c)])
#define gdImageBlue(im, c) ((im)->trueColor ? gdTrueColorGetBlue(c) : \
	(im)->blue[(c)])
#define gdImageAlpha(im, c) ((im)->trueColor ? gdTrueColorGetAlpha(c) : \
	(im)->alpha[(c)])
#define gdImageGetTransparent(im) ((im)->transparent)
#define gdImageGetInterlaced(im) ((im)->interlace)

/* These macros provide direct access to pixels in
	palette-based and truecolor images, respectively.
	If you use these macros, you must perform your own
	bounds checking. Use of the macro for the correct type
	of image is also your responsibility. */
#define gdImagePalettePixel(im, x, y) (im)->pixels[(y)][(x)]
#define gdImageTrueColorPixel(im, x, y) (im)->tpixels[(y)][(x)]

/* I/O Support routines. */

  BGD_EXPORT gdIOCtx *gdNewFileCtx (FILE *);
  /* If data is null, size is ignored and an initial data buffer is
    allocated automatically. NOTE: this function assumes gd has the right 
    to free or reallocate "data" at will! Also note that gd will free 
    "data" when the IO context is freed. If data is not null, it must point
    to memory allocated with gdMalloc, or by a call to gdImage[something]Ptr.
    If not, see gdNewDynamicCtxEx for an alternative. */
  BGD_EXPORT gdIOCtx *gdNewDynamicCtx (int size, void *data);
  /* 2.0.21: if freeFlag is nonzero, gd will free and/or reallocate "data" as
    needed as described above. If freeFlag is zero, gd will never free 
    or reallocate "data," which means that the context should only be used
    for *reading* an image from a memory buffer, or writing an image to a
    memory buffer which is already large enough. If the memory buffer is
    not large enough and an image write is attempted, the write operation
    will fail. Those wishing to write an image to a buffer in memory have
    a much simpler alternative in the gdImage[something]Ptr functions. */
  BGD_EXPORT gdIOCtx *gdNewDynamicCtxEx (int size, void *data, int freeFlag);
  BGD_EXPORT gdIOCtx *gdNewSSCtx (gdSourcePtr in, gdSinkPtr out);
  BGD_EXPORT void *gdDPExtractData (struct gdIOCtx *ctx, int *size);

#define GD2_CHUNKSIZE           128
#define GD2_CHUNKSIZE_MIN	64
#define GD2_CHUNKSIZE_MAX       4096

#define GD2_VERS                2
#define GD2_ID                  "gd2"

#define GD2_FMT_RAW             1
#define GD2_FMT_COMPRESSED      2

/* Image comparison definitions */
   BGD_EXPORT int gdImageCompare (gdImagePtr im1, gdImagePtr im2);

#define GD_CMP_IMAGE		1	/* Actual image IS different */
#define GD_CMP_NUM_COLORS	2	/* Number of Colours in pallette differ */
#define GD_CMP_COLOR		4	/* Image colours differ */
#define GD_CMP_SIZE_X		8	/* Image width differs */
#define GD_CMP_SIZE_Y		16	/* Image heights differ */
#define GD_CMP_TRANSPARENT	32	/* Transparent colour */
#define GD_CMP_BACKGROUND	64	/* Background colour */
#define GD_CMP_INTERLACE	128	/* Interlaced setting */
#define GD_CMP_TRUECOLOR	256	/* Truecolor vs palette differs */

/* resolution affects ttf font rendering, particularly hinting */
#define GD_RESOLUTION           96	/* pixels per inch */

/* newfangled special effects */
#include "gdfx.h"

#ifdef __cplusplus
}
#endif

#endif				/* GD_H */
